/*******************
** Eldarea MUDLib **
********************
**
** global/daemon/cron.c
**
** CVS DATA
** $Date: 2001/05/05 10:41:46 $
** $Revision: 1.2 $
**
** mud cron daemon
**
** CVS History
**
** $Log: cron.c,v $
** Revision 1.2  2001/05/05 10:41:46  elatar
** removed some bugs with system crontabs
**
** Revision 1.1  2001/05/05 10:16:58  elatar
** initial revision in cvs
**
**
**
*/

/* MUD Cron Daemon
 * 
 * Version      0.1.7
 * Date         17.08.2000
 * Last edited  08.05.2001
 * Author       Nicolai Ehemann (Elatar@Eldarea)
 *
 * Much thanks to Anatol@Silberland, as he provided many problems
 * and the corresponding solutions in his cron daemon.
 *
 */

#pragma strict_types

/*
 * inherit an editor for crontab editing...
 * nedit isnt very good, but the only one we have
 */
inherit "/mail/nedit";

#include <wizlevels.h>
#include <config.h>
#include <daemon/channel.h>
#define NEED_CRON_PROTOTYPES
#include <cron.h>

//#define DB(x)
#define DB(x) if (find_player("elatar")) tell_object(find_player("elatar"),x+"\n")

/** all pending  jobs of the day */
static mapping jobs;
/** urgent jobs. these will be done asynchronous */
mixed * urgent_jobs;
/** all valid crontabs */
static mapping tabs;
/** last minute jobs were executed */
int last_executed;

nomask void create()
{
  if(clonep(this_object())) 
    return;
  seteuid(getuid());
  printf("Starting cron...\n");
  printf("Reloading job queue\n");
  restore_object(load_name());
  if (!pointerp(urgent_jobs))
    urgent_jobs = ({});
  ReadCrontabs();
  printf("Adding system entry at 0:00 hours\n");
  TryAddJobToList(SYSTEM_JOB,getuid());
  printf("Starting first call_out\n");
  StartNextCallout();
  printf("done.\n");
  VOICEMASTER->send("Cron",this_object(),"cron loaded.");
}

/*
 * prevent cron from beeing shadowed
 *
 */
nomask public int query_prevent_shadow(object po) { return 1; }

/*
 * channel master name for cron channel
 *
 */
nomask public string name() { return dtime()[19..]; }

/*
 * check if channel action is valid
 * (no messages in first implementation)
 *
 */
int check(string ch, object pl, string cmd, string txt)
{
  if(ch != "Cron") 
    return 0;
  if(objectp(pl) && query_once_interactive(pl) && query_wiz_level(pl) > 1)
  {
    switch (cmd)
    {
      case C_FIND:
      case C_LIST:
      case C_JOIN:
      case C_LEAVE:
        return 1;
      case C_SEND:
    }
  }
  return 0;
}

/*
 * the only public function to modify internals
 * this should be called directly through the 'crontab'
 * command in the player object
 *
 */
nomask public int cmd_crontab(string * args)
{
  string uid, dummy;
  
  if ( !this_player()
    || !IS_WIZARD(this_player())
    || !previous_object()
    || previous_object()!=this_player()
     )
  {
    CError("Invalid call to 'cmd_crontab'");
    return 1;
  }
  
  // determine uid... archs are allowed to use -u option
  if (args[0]=="-u")
  { 
    if (sizeof(args)<2)
    {
      return -1;
    }
    if (!IS_ARCH(this_player()))
    {
      notify_fail("must be privileged to use -u\n");
      return 0;
    }
    if (!IS_WIZARD(args[1]) && args[1]!="root")
    {
      notify_fail("user unknown "+args[1]+"\n");
      return 0;  
    }
    uid=args[1];
    args=args[2..];
  }
  else
  {
    uid=getuid(this_player());  
  }
  
  if (!sizeof(args))
  {
    return -3;
  }
  if (sizeof(args)>1)
  {
    return -2;
  }
  if (args[0]=="-l")
  { // list crontab
    write(ListCrontab(uid));
  }
  else if (args[0]=="-r")
  {
    if (member(tabs,uid))
    { // remove crontab
      RemoveJobs(uid);
      rm(CRONTABS"/"+uid);
      tabs=m_delete(tabs,uid);
      CLog("Crontab for "+uid+" removed by "+getuid(this_player())+".");
      write("crontab: removed crontab\n");
    }
    else
    {
      write("crontab: no crontab for "+uid+"\n");
    }
  }
  else if (args[0]=="-e")
  { // edit crontab; make new if none present
    if (file_size(CRONTABS"/"+uid)<1)
    {
      write("New crontab for "+uid+"\n"
            "** oder . wenn Ihr fertig seid, ~q zum "
                "Abbrechen, ~h fuer eine Hilfsseite.\n");
      nedit("PostEditCrontab","",({"",uid,getuid(this_player())}));
    }
    else
    {
      write("Reading crontab for "+uid+"\n"
            "** oder . wenn Ihr fertig seid, ~q zum "
                "Abbrechen, ~h fuer eine Hilfsseite.\n");
      dummy=read_file(CRONTABS"/"+uid);
      nedit("PostEditCrontab",dummy,({dummy,uid,getuid(this_player())}));
    }
  }
  else if (sscanf(args[0],"-%s",dummy)==1)
  { 
    return -4;
  }
  else
  { // argument should be a file to replace
    switch(file_size(args[0]))
    {
      case -2:
        write(args[0]+" is a directory\n");
        break;
      case -1:
      case 0:
        write(args[0]+": file not found\n");
        break;
      default:
        // read the file and put it into crontab
        if (member(tabs,uid))
          rm(CRONTABS"/"+uid);
        UpdateCrontab(uid);
        StartNextCallout();
        CLog("Crontab for "+uid+" installed by "+getuid(this_player())+".");
        write("crontab: crontab "+args[0]+" installed\n");
        break;
    }
  }
  return 1;
}

/*
 * save crontab after editing, if changed
 *
 */
nomask static void PostEditCrontab(string ntab, string * args)
{
  if (!ntab || ntab==args[0])
  {
    write("crontab: no changes made to crontab\n");
  }
  else
  {
    rm(CRONTABS"/"+args[1]);
    write_file(CRONTABS"/"+args[1],ntab);
    UpdateCrontab(args[1]);
    StartNextCallout();
    CLog("Crontab for "+args[1]+" installed by "+args[2]+".");
    write("crontab: installing new crontab\n");
  }
}

/*
 * Write message to log
 *
 */
nomask static void CLog(string line)
{
  log_file("daemon/cron",sprintf("%s %s\n",ctime()[4..],line)); 
}

/*
 * Print and log Error message
 *
 */
nomask static varargs void CError(string err, int silent)
{
  if (!silent)
    raise_error(err);
  else
    VOICEMASTER->send("Cron",this_object(),sprintf("Error: %s",err));
  log_file("daemon/cron.err",sprintf("%s %s\n",ctime()[4..],err));
}

/*
 * Return crontab of uid
 *
 */
nomask static string ListCrontab(string uid)
{
  if (!member(tabs,uid))
    return "crontab: no crontab for "+uid+"\n";
  
  return read_file(CRONTABS"/"+uid);
}

/*
 * Start the next call_out in cron job list
 *
 */
nomask static void StartNextCallout()
{
  int * ind, *this_time, i;
  
  // remove pending callout
  while (remove_call_out("StartNextCallout")!=-1);
  
  this_time=GetDayData();
  
  // sort job indices
  ind=sort_array(m_indices(jobs),#'>);
  
  /*
  // all jobs older than DISCARD_MINUTES are discarded
  while ( i<sizeof(ind) 
       && ind[i] < this_time[CTI_MINUTES]
                  +this_time[CTI_HOURS]*60
                  -DISCARD_MINUTES)
  {
    efun::m_delete(jobs,ind[i++]);
  }
  */
  // do not add a job again
  while (i<sizeof(ind) && ind[i] <= last_executed)
    i++;

  // add all jobs up to now to urgent_jobs (to be done)
  while ( i<sizeof(ind)
       && ind[i]%(24*60) <= this_time[CTI_MINUTES]+this_time[CTI_HOURS]*60 )
  { 
    last_executed = ind[i];
    urgent_jobs=({})+jobs[ind[i]];
    efun::m_delete(jobs,ind[i++]);
  }
  save_object(load_name());

  if (sizeof(urgent_jobs))
    DoJobs();
  if (i<sizeof(ind))
    return;
    
  // we are very close to the next, call_out directly
  if (ind[i]<=1+this_time[CTI_MINUTES]+this_time[CTI_HOURS]*60)
  {
    DB("very close, direct call");
    call_out("StartNextCallout",
      ind[i]*60-
        (this_time[CTI_SECONDS]+
         this_time[CTI_MINUTES]*60+
         this_time[CTI_HOURS]*60*60));
  }
  // we are close, got to get closer
  else if (ind[i]<=5+this_time[CTI_MINUTES]+this_time[CTI_HOURS]*60)
  {
    DB("close");
    call_out("StartNextCallout",
      (ind[i]-1)*60-
        (this_time[CTI_MINUTES]*60+
         this_time[CTI_HOURS]*60*60));
  }
  // not as close, get closer
  else if (ind[i]<=15+this_time[CTI_MINUTES]+this_time[CTI_HOURS]*60)
  {
    DB("get closer");
    call_out("StartNextCallout",
      (ind[i]-5)*60-
        (this_time[CTI_MINUTES]*60+
         this_time[CTI_HOURS]*60*60));
  }
  // wait until closer
  else
  {
    DB("wait");
    call_out("StartNextCallout",
      (ind[i]-15)*60- // minute to start cl
        (this_time[CTI_MINUTES]*60+   // already passed minutes
         this_time[CTI_HOURS]*60*60)); // already passed hours
  }
}

/*
 * Do the jobs in the urgent job list (asynchronous)
 *
 */
nomask static void DoJobs()
{
  int i;
  
  // remove pending call_outs (there wont be any)
  while(remove_call_out("DoJobs")!=-1);
  
  // do a maximum of JOBS_PER_CYCLE jobs per cycle
  for (i=0;i<sizeof(urgent_jobs)&&i<JOBS_PER_CYCLE;i++)
  {
    TryDoJob(urgent_jobs[i]);
  }
  
  urgent_jobs=urgent_jobs[i..];
  if (sizeof(urgent_jobs))
    call_out("DoJobs",2);
  save_object(load_name());
}

/*
 * Check if job is allowed and execute if so
 *
 */
nomask static void TryDoJob(mixed * job)
{
  object obj;
  string err;
  
  // check if file exists
  if (file_size(job[CJ_OBJ]+".c")<1)
    return CError(sprintf(
      "Job: %s (%s()) by uid %s failed: no such file.",
      job[CJ_OBJ],job[CJ_FUN],job[CJ_UID]),1);
  
  // check if file is loadable if not still loaded
  if (!find_object(job[CJ_OBJ]) && err=catch(load_object(job[CJ_OBJ])))
    return CError(sprintf(
      "Job: %s (%s()) by uid %s failed: object could not be loaded (%s).",
      job[CJ_OBJ],job[CJ_FUN],job[CJ_UID],err),1);
  
  // check if uid is allowed to read here
  if (!MASTER->valid_read(job[CJ_OBJ], job[CJ_UID], job[CJ_FUN], obj))
    return CError(sprintf(
      "Job: %s (%s()) by uid %s failed: permission denied.",
      job[CJ_OBJ],job[CJ_FUN],job[CJ_UID]),1);
  
  // call, and handle possible error
  if (err=catch(call_other(job[CJ_OBJ],job[CJ_FUN])))
    return CError(sprintf(
      "Job: %s (%s()) by uid %s failed: Error %s.",
      job[CJ_OBJ],job[CJ_FUN],job[CJ_UID],err),1);
  VOICEMASTER->send("Cron",this_object(),sprintf("Job: %s (%s()) by %s done.",
                    job[CJ_OBJ],job[CJ_FUN],job[CJ_UID]));
}

/*
 * Update one crontab, check if valid and
 * add jobs to today list
 *
 */
nomask static void UpdateCrontab(string uid)
{
  string * crontab;
  int i;
  
  RemoveJobs(uid);
  if (!IS_WIZARD(uid) && uid!=ROOTID && uid!="root")
  { // Only Wizards are allowed to cron
    rm(CRONTABS"/"+uid);
    tabs=m_delete(tabs,uid);
    CLog("Crontab for "+uid+" removed. (wizlevel too low)");
    return;
  }
  if (member(tabs,uid))
  {
    tabs[uid,CT_FILEDATE]=file_date(CRONTABS"/"+uid);
    tabs[uid,CT_CRONJOBS]=({});
  }
  else
    tabs+=([uid:file_date(CRONTABS"/"+uid);({})]);
  crontab=explode(read_file(CRONTABS"/"+uid),"\n");
  DB(sprintf("%O",crontab));
  for (i=sizeof(crontab);i-->0;)
  {
    if (strlen(crontab[i]) && crontab[i][0]!='#');
      AddJob(crontab[i],uid);
  }
}

/*
 * Read all crontabs and update them
 * (to be done at startup time and at 
 * 24.00 hours by system job)
 *
 */
nomask static void ReadCrontabs()
{
  string * crontabs;
  int i;
  
  tabs=([]);
  jobs=([]);
  printf("Reading crontabs in "CRONTABS"/ ");
  crontabs = get_dir(CRONTABS"/*")-({"CVS",".",".."});;
  DB(sprintf("%O",crontabs));
  for (i=sizeof(crontabs);i-->0;)
  {
    printf(".");
    UpdateCrontab(crontabs[i]);  
  }
  printf("\n");
}

/*
 * Check if one value of a cronjob is correct
 * (syntax and bounds)
 *
 */
nomask mixed * EvalValue(string val, int min, int max)
{
  int d, dd, i;
  string s,ss;
  mixed * ret, *cnt;
  
  if (val=="*")
    return ({({min,max})});
  
  for (i=0;i<sizeof(cnt=explode(val,","));i++)
  {  
    if (sscanf(cnt[i],"%d-%s",d,s)==2)
    { // range spec
      if (sscanf(s,"%d%s",dd,ss)==2 && d<dd && d>=min && dd<=max)
      { // correct range format
        if (pointerp(ret))
        {
          if ( ( pointerp(ret[<1]) 
              && d>ret[<1][1] ) 
            || ( !pointerp(ret[<1]) 
              && d>ret[<1] ) )
          {
            if (d+1==dd)
              ret+=({d,dd});
            else
              ret+=({({d,dd})});
          }
          else
            return 0;
        }
        else
        {
          if (d+1==dd)
            ret=({d,dd});
          else
            ret=({({d,dd})});
        }
      }
      else
        return 0;
    }  
    else if (sscanf(cnt[i],"%d%s",d,s)==2 && !strlen(s))
    { // single spec
      if (d>=min && d<=max)
      { // correct single format
        if (pointerp(ret))
        {
          if ( ( pointerp(ret[<1]) 
              && d>ret[<1][1] ) 
            || ( !pointerp(ret[<1]) 
              && d>ret[<1] ) )
            ret+=({d});
          else 
            return 0;
        }
        else
          ret=({d});
      }
      else
        return 0;      
    }
    else
    { // bullshit
      return 0;
    }
  }
  return ret;
}

/*
 * add one job to the cron jobs table
 *
 */
nomask static void AddJob(string tabline, string uid)
{
  string min,hou,day,mon,wda,obj,fun;
  mixed * job, val;
  
  if (sscanf(tabline,"%s%t%s%t%s%t%s%t%s%t%s%t%s",min,hou,day,mon,wda,obj,fun)!=7)
  {
    CError("Invalid format in "CRONTABS"/"+uid+".",1);
    return;
  }
    
  job=allocate(7);
  if (!val=EvalValue(min,0,59))
  {
    CError("Invalid value in "CRONTABS"/"+uid+".",1);
    return;
  }
  job[CTI_MINUTES]=val;
  if (!val=EvalValue(hou,0,23))
  {
    CError("Invalid value in "CRONTABS"/"+uid+".",1);
    return;
  }
  job[CTI_HOURS]=val;
  if (!val=EvalValue(day,0,31))
  {
    CError("Invalid value in "CRONTABS"/"+uid+".",1);
    return;
  }
  job[CTI_DAYS]=val;
  if (!val=EvalValue(mon,0,11))
  {
    CError("Invalid value in "CRONTABS"/"+uid+".",1);
    return;
  }
  job[CTI_MONTHS]=val;
  if (!val=EvalValue(wda,0,6))
  {
    CError("Invalid value in "CRONTABS"/"+uid+".",1);
    return;
  }
  job[CTI_WEEKDAYS]=val;
  if (!stringp(obj))
  {
    CError("Invalid object name in "CRONTABS"/"+uid+".",1);
    return;
  }
  job[CTI_OBJECT]=obj;
  if (!stringp(fun))
  {
    CError("Invalid function name in "CRONTABS"/"+uid+".",1);
    return;
  }
  job[CTI_FUNCTION]=trim(fun);
  DB(sprintf("%O",job));
  tabs[uid,CT_CRONJOBS]+=({job});
  TryAddJobToList(job, uid);
}

/*
 * Add the job to the list, if it is the correct day
 * and the time isnt already over
 *
 */
nomask static void TryAddJobToList(mixed * job, string uid)
{
  int i, j, k, l, * this_time, *mins;
  
  // check if month, weekday and day are correct
  this_time=GetDayData();
  
  for (i=CTI_DAYS;i<=CTI_WEEKDAYS;i++)
  {
    DB(sprintf("%O %O",this_time[i],job[i]));
    if (!CheckRange(this_time[i],job[i]))
      return;
  }
  
  // ok, insert into list
  // therefore get the correct minute(s)
  mins=({});
  for (i=0;i<sizeof(job[CTI_HOURS]);i++)
  {
    if (pointerp(job[CTI_HOURS][i]))
    {
      for (k=job[CTI_HOURS][i][0];k<=job[CTI_HOURS][i][1];k++)
      {
        for (j=0;j<sizeof(job[CTI_MINUTES]);j++)
        {
          if (pointerp(job[CTI_MINUTES][j]))
          {
            for (l=job[CTI_MINUTES][i][0];l<=job[CTI_MINUTES][i][1];l++)
            {
              mins+=({k*60+l});
            }
          }
          else
            mins+=({k*60+job[CTI_MINUTES][j]});
        }
      }
    }
    else
    {
      for (j=0;j<sizeof(job[CTI_MINUTES]);j++)
      {
        if (pointerp(job[CTI_MINUTES][j]))
        {
          for (l=job[CTI_MINUTES][j][0];l<=job[CTI_MINUTES][j][1];l++)
          {
            mins+=({job[CTI_HOURS][l]*60+l});
          }
        }
        else
          mins+=({job[CTI_HOURS][l]*60+job[CTI_MINUTES][j]});
      }      
    }
  }

  // remove all entrys before this_time
  for(i=0;i<sizeof(mins) 
       && mins[i]<this_time[CTI_MINUTES]+this_time[CTI_HOURS]*60;i++);
  
  DB(sprintf("%O",mins[i..]));
  for (i;i<sizeof(mins);i++)
  {
    if (member(jobs,mins[i]))
      jobs[mins[i]]+=({({uid,job[CTI_OBJECT],job[CTI_FUNCTION]})});
    else
      jobs[mins[i]]=({({uid,job[CTI_OBJECT],job[CTI_FUNCTION]})});
  }
}

/*
 * remove all pending jobs of one uid
 *
 */
nomask static void RemoveJobs(string uid)
{
  DB(sprintf("%O",jobs));
  walk_mapping(jobs,lambda(
    ({'min,'jl}),
      ({#'=,'jl,({#'filter,'jl,
        ({#'lambda,quote(({'job,'eid})),quote(({#'!=,({#'[,'job,CJ_UID}),'eid}))})
        ,uid})})));
  jobs = filter(jobs, lambda(({'min,'jl}),({#'sizeof,'jl})));
  DB(sprintf("%O",jobs));
}

/*
 * Return an array holding the actual time in the same
 * order as in crontab 
 *
 */
nomask static int * GetDayData()
{
  string s_wda,s_mon;
  int wda,day,mon,yea,hou,min,sec;

  sscanf(dtime(),"%s, %d. %s %d, %d:%d:%d",s_wda,day,s_mon,yea,hou,min,sec);
  wda=wday_to_int(s_wda);
  mon=month_to_int(s_mon);
  return ({min,hou,day,mon,wda,sec});
}


/*
 * Check if the value is in crontab range
 *
 */
nomask static int CheckRange(int val, mixed * range)
{
  int i;
  
  for (i=0;i<sizeof(range);i++)
  {
    if (pointerp(range[i]))
    {
      // range spec, check bounds
      if (val>=range[i][0] && val<=range[i][1])
        return 1;
    }
    // value check
    else if (val==range[i])
      return 1;
  }
  
  // sorry, no match found
  return 0;
}

/*
 * Clear memory and update all crontabs from files
 * This function is called at 24.00 hours by the
 * system job.
 *
 */
nomask static void UpdateCrontabs()
{
  last_executed = -1;
  save_object(load_name());
  ReadCrontabs();
  StartNextCallout();
}