/*******************
** Eldarea MUDLib **
********************
**
** global/handler/life.c - global life event handler
**
** CVS DATA
** $Date: 2000/12/18 14:53:19 $
** $Revision: 1.3 $
**
** handled events:
**
** ET_DEATH     livings dieing
** ET_EXHAUST   exhaustion
**
** CVS History
**
** $Log: life.c,v $
** Revision 1.3  2000/12/18 14:53:19  elatar
** ET_DEATH event now provides messages
** ET_EXHAUS event implemented
**
** Revision 1.2  2000/12/15 15:33:09  elatar
** P_VICTIM_DIE_FLAGS inserted
**
** Revision 1.1  2000/12/15 15:26:01  elatar
** initial revision of life event handler
** handled event: ET_DEATH
**
*/

#pragma strict_types

#define DEBUG

#include <events.h>
#include <living/life.h>
#include <properties.h>
#include <moving.h>
#include <wizlevels.h>
#include <corpse.h>

void receive_event(mixed data, string type, int prio, int mode);

void create() 
{
  if(clonep()) 
  { 
    destruct(this_object()); 
    return; 
  }

  seteuid(getuid());
  listen_event(ET_DEATH,     EPRIO_DEF_HANDLE, #'receive_event);
  listen_event(ET_EXHAUST,   EPRIO_DEF_HANDLE, #'receive_event);
  set_global_listener(ET_DEATH, 1);
  set_global_listener(ET_EXHAUST, 1);
}

static void create_kill_log_entry(object victim, string killer)
{
  string tmp;
  int level,lost_exp;

  tmp=dtime(time());
  if ((level=(int)victim->QueryProp(P_LEVEL))<20)
    lost_exp = (int)victim->QueryProp(P_XP)/3;
  else
    lost_exp = (int)victim->QueryProp(P_XP)/(level-17);
  log_file("KILLS",
	   tmp[5..11]+" "+tmp[19..23]+" "+
	   capitalize(getuid(victim))+
	   "("+level+","+(lost_exp/1000)+") "+
	   killer+"\n");
}

void transfer_all(object from, mixed dest)
{
  object *ob;

  ob = all_inventory(from);
  while(sizeof(ob))
  {
    if ( ! ob[0]->QueryProp( P_NEVERDROP ) )
	    ob[0]->move(dest,M_SILENT|M_NOCHECK|M_MOVE_ALL);
    ob = ob[1..];
  }
}

void remove_all(object from)
{
  object *ob;

  ob = all_inventory(from);
  while(sizeof(ob))
  {
    if ( !ob[0]->QueryProp( P_NEVERDROP ) )
      ob[0]->deep_remove();
    ob = ob[1..];
  }      
}

void die(mapping data)
{
  object corpse, victim;
  mixed killer;
  int flags;
  
  killer = data[E_DEATH_KILLER];

  if ( extern_call() && (previous_object()!=killer || !living(killer)))
    raise_error(
      sprintf("EVENT_HANDLER %s illegal call from %s\n",
        load_name(),load_name(previous_object())));
        
  victim = data[E_DEATH_VICTIM];
  flags = data[E_DEATH_FLAGS];
        
  if (victim->QueryProp(P_POISON))
    victim->SetProp(P_POISON, 0);
    
  if(query_once_interactive(victim) && IS_LEARNING(victim)) 
  {
    // this message is 'silent' :)
    tell_object(victim,
      "Wenn Ihr nicht unsterblich waert, waere dies gerade Euer "
      "Ende gewesen.\n");
    handle_event(IMMORTAL);
    return;
  }

  if (flags&DIE_DIRECT) 
  {
    string tmp;
    
    tmp=explode(killer?load_name(killer):"unbekannt","#")[0];
	  if (strlen(tmp)>33) 
	    tmp="..."+tmp[<31..];
	  tmp+=" (direkt)";
    create_kill_log_entry(victim, tmp+" ("+getuid(killer)||"unbekannt"+")");
  }
  
  if (!(flags|DIE_SILENT))
  {
    string msg;
    
    msg = data[E_DEATH_RMSG];
    msg = implode(explode(msg,"&Name"),(string) victim->Name());
    msg = implode(explode(msg,"&name"),(string) victim->name());
    
    tell_room(environment(victim),break_string(msg),({victim}));
    
    if (stringp(msg=data[E_DEATH_PMSG]))
      tell_object(victim,break_string(msg));
  }

  if (!victim->QueryProp(P_EP_FUNC) && killer && 
      query_once_interactive(killer) && !(flags|DIE_NOEP))
    killer->GiveEP(); 

  victim->SetProp(P_HP, -1);

  if (flags|DIE_NOCORPSE)
  {
    if (flags|DIE_REMOVE_INV)
      remove_all(victim);
    else
      transfer_all(victim, environment(victim));
  }
  else 
  {
    corpse = clone_object(data[E_DEATH_CORPSE]);
    data[E_DEATH_CORPSE_OBJ] = corpse;
      
    corpse->SetProp( P_KILLER, (killer?load_name(killer):0) );
    corpse->SetProp( P_VICTIM, load_name(victim));
    if(query_once_interactive(victim)) 
      corpse->SetProp( P_VICTIM_TYPE, 1);
    if(victim->QueryProp(P_NPC)) 
      corpse->SetProp( P_VICTIM_TYPE, 2);
    corpse->SetProp( P_VICTIM_NAME, victim->QueryProp(P_NAME));      
    corpse->SetProp( P_VICTIM_PLURAL, victim->QueryProp(P_PLURAL));      
    corpse->SetProp( P_VICTIM_ARTICLE, victim->QueryProp(P_ARTICLE));
    corpse->SetProp( P_VICTIM_GENDER, victim->QueryProp(P_GENDER));
    corpse->SetProp( P_VICTIM_LEVEL, victim->QueryProp(P_LEVEL));
    corpse->SetProp( P_VICTIM_REAL_STATS, victim->QueryProp(P_REAL_STATS));
    corpse->SetProp( P_VICTIM_MAX_HP, victim->QueryProp(P_MAX_HP));
    corpse->SetProp( P_VICTIM_XP, victim->QueryProp(P_XP));
    corpse->SetProp( P_VICTIM_HANDS, victim->QueryProp(P_HANDS));
    corpse->SetProp( P_VICTIM_MAX_HANDS, victim->QueryProp(P_MAX_HANDS));
    corpse->SetProp( P_VICTIM_ARMOURS, victim->QueryProp(P_ARMOURS));
    corpse->SetProp( P_VICTIM_WEAPONS, victim->QueryProp(P_WEAPONS));
    corpse->SetProp( P_VICTIM_BODY, victim->QueryProp(P_BODY));
    corpse->SetProp( P_VICTIM_RACE, victim->QueryProp(P_RACE));
    corpse->SetProp( P_VICTIM_ALIGN, victim->QueryProp(P_ALIGN));
    corpse->SetProp( P_VICTIM_WEIGHT, victim->QueryProp(P_WEIGHT));
    corpse->SetProp( P_VICTIM_AGGRESSIVE, victim->QueryProp(P_AGGRESSIVE));
    corpse->SetProp( P_VICTIM_DIE_FLAGS, flags);
    // Etwas komische Methode rauszufinden, obs ein Container ist,
    // aber find was besseres ;*)
    if (flags|DIE_REMOVE_INV)
      remove_all(victim);
    else if (function_exists("make_invlist", corpse)) 
      transfer_all(victim, corpse);
    else 
      transfer_all(victim, environment(victim));
    corpse->move(environment(victim),M_NOCHECK);
  }
  if(!victim->second_life(corpse)) 
  {
    victim->remove();
    return handle_event(DEAD);
  }
  victim->SetProp(P_ALCOHOL, 0);
  victim->SetProp(P_FOOD, 0);
  victim->SetProp(P_DRINK, 0);
  victim->SetProp(P_FROG, 0);
  
  return handle_event(DEAD);  
}

static void exhaust(object snd, mapping data)
{
  int exp, ex;
    
  ex = (int) data[E_EXHAUST_EXHAUST];
    
  exp=(int) snd->QueryProp(P_EXP);
  if (exp<ex)
  {
    snd->SetProp(P_EXP,0);
    handle_event(exp);
  }
  else
  {
    exp-=ex;
    snd->SetProp(P_EXP,exp);
    handle_event(ex);
  }
  return;
}

void receive_event(mixed data, string type, int prio, int mode)
{
  mixed snd;
  
#ifdef DEBUG
  if(this_player() && this_player()->QueryProp("debug_events"))
    printf("EVENT_HANDLER: %s <%s>\nMODE: %d\nDATA: %O\n",
      load_name() ,type, mode, data);
#endif

  if(load_name(previous_object())!=EVENTD)
    raise_error(
      sprintf("EVENT_HANDLER %s: illegal call from %s\n", 
        load_name(), load_name(previous_object())));

  if(!mappingp(data))
    raise_error(
      sprintf("EVENT_HANDLER %s: data mapping empty\n", load_name()));

  if(data[E_HANDLED]) 
    return; 

  if(!objectp(snd=data[E_SENDER]))
    raise_error(
      sprintf("EVENT_HANDLER %s: no event sender specified\n", load_name()));

  switch (type)
  {
    case ET_DEATH:
      die(data);
      break;
    case ET_EXHAUST:
      exhaust(snd, data);
      break;
    default:
      raise_error(
        sprintf("EVENT_HANDLER %s: unexpected event type %s from %s",
          load_name(),type,load_name(previous_object())));
      break;
  }
}
