/*******************
** Eldarea MUDLib **
********************
**
** global/service/clock.c - mud clock
**
** CVS DATA
** $Date: 2001/02/08 16:09:14 $
** $Revision: 1.3 $
**
** Diese Uhr arbeitet mit call_outs. Leider scheint der call_out-Mechanismus
** wirklich alles andere als genau zu sein, in einer Stunde kann er um mehrere
** Minuten zu langsam sein. Daher treffe ich geeignete Korrekturmassnahmen
**
** Autor: Troy@Wunderland
** Multiple Zeitzonen: Fiona@Wunderland
**
** CVS History
**
** $Log: clock.c,v $
** Revision 1.3  2001/02/08 16:09:14  elatar
** timezones and handling completely rewritten, ET_CHANGE_DAYTIME event implemented
**
** Revision 1.2  2001/01/17 15:08:38  elatar
** new header path adapted
**
** Revision 1.1  2001/01/17 14:56:02  elatar
** renamed uhr.c to clock.c
**
** Revision 1.2  2000/12/01 15:31:15  elatar
** clock messages slightly changed
**
** Revision 1.1.1.1  1999/11/05 12:30:43  elatar
** Preparing mudlib for cvs control
**
**
*/

#pragma strong_types

#include <properties.h>
#include <wizlevels.h>
#include <config.h>
#include <events.h>

#define NEED_PROTOTYPES
#include <clock.h>

// Enthaelt die Zeitzonen jeweils als Array mit Minutenangaben und Status
mapping zones;

// Prototypen interner Funktionen
void WaitForFullHour();
static void ChangeOccurs(zone_t zone, int seq, int state);
static void WaitForChange();
static void StartWait();
varargs private zone_t get_tz(object pl);
private time_t* get_nexts(zone_t zone);
private void shoutout(string txt);
varargs void notify_all(zone_t zone);

void create() 
{
	int i;
	object pl;
	if (clonep()) 
	{
		destruct(this_object());
		return;
	}
	zones=MEMORY->get("zones");
	if (!mappingp(zones) || !sizeof(zones)) 
	{
		MEMORY->put("zones", ZONEDEFS);
		zones=MEMORY->get("zones");
	}
	WaitForFullHour();
	StartWait();
}


// *** Funktionen, die den Tageszustand widerspiegeln ***

// Liefert den Tageszustand des aufrufenden Objektes als String
// Argument kann 0 (Standard) spezielles Obj oder Zonennummer in
// einelementigem Array sein
varargs public string GetLighting(mixed mix) {
	if (IsNight(mix)) return "Nacht";
	return "Tag";
}

// Liefert den Tageszustand des aufrufenden Objektes als int
// Argument kann 0 (Standard) spezielles Obj oder Zonennummer in
// einelementigem Array sein
varargs public int IsNight(mixed mix)
{
	zone_t tz;
	time_t* x;
	if (pointerp(mix)) tz=mix[0];
	else tz=get_tz(mix);
	x=get_nexts(tz);
	if (x[1]>x[0]) return 1;
	return 0;
}

// Liefert den Status der Zeitzone (fuer das aufrufende Objekt)
// Argument kann 0 (Standard) spezielles Obj oder Zonennummer in
// einelementigem Array sein
varargs public int QueryState(mixed mix)
{
	zone_t tz;
	if (pointerp(mix)) tz=mix[0];
	else tz=get_tz(mix);
	return zones[tz,STATE]-1;
}

// Liefert Zeitpunkt des naechsten Tages (fuer das aufrufende Objekt)
// Argument kann 0 (Standard) spezielles Obj oder Zonennummer in
// einelementigem Array sein
varargs public time_t QueryDay(mixed mix)
{
	if (pointerp(mix)) return get_nexts(mix[0])[0];
	return get_nexts(get_tz(mix))[0];
}

// Liefert Zeitpunkt der naechsten Nacht (fuer das aufrufende Objekt)
// Argument kann 0 (Standard) spezielles Obj oder Zonennummer in
// einelementigem Array sein
varargs public time_t QueryNight(mixed mix)
{
	if (pointerp(mix)) return get_nexts(mix[0])[1];
	return get_nexts(get_tz(mix))[1];
}

// Liefert Zeitpunkt des naechsten Wechsels (fuer das aufrufende Objekt)
// Argument kann 0 (Standard) spezielles Obj oder Zonennummer in
// einelementigem Array sein
varargs public time_t QueryChange(mixed mix)
{
	time_t* x;
	
	if (pointerp(mix)) x=get_nexts(mix[0]);
	else x=get_nexts(get_tz(mix));
	if (x[1]>x[0]) return x[0];
	return x[1];
}

varargs public min_t QueryDayLength(mixed mix)
{
  zone_t tz;
  
  tz=get_tz(mix);
  
  return zones[tz, DAY]+zones[tz, DAWN]+zones[tz, NIGHT]+zones[tz, DUSK];
}

// *** Erzeugt Glockengelaeut zur vollen RL-Stunde ***

// Starten des call_outs
static void WaitForFullHour()
{
	int i;
	while (remove_call_out("HourReached")!=-1);
	i=3600-time()%3600; // Sekunden bis zur naechsten vollen Stunde
	if (i>300) i=i-300; /* Bei lags werden die call_outs langsamer ... */
	else i=0;	  /* Also lieber Verspaetung einplanen ... */
	call_out("HourReached",i);
}

// Wartet auf volle RL-Stunde zum Zeitansagen
static void HourReached()
{
	int i;
	int min;
  
	i=get_day_ticks(time());
	min=(i%3600)/60;
	i/=3600;
	if (min>53) {
		call_out("HourReached", 30); // Nah dran .. in 30 sek nochmal
		return;
	}
	if (min>10) {   // Verpasst .. nicht mehr shouten
		WaitForFullHour();
		return;
	}
	if (min>1) {
		// Hope this will never happen.
		shoutout(sprintf("Oops! Wir haben vergessen, die Zeit anzusagen ... "
			"es ist jetzt %02d:%02d Uhr.\n", i, min));
		WaitForFullHour();
		return;
	}
	shoutout("Es ist jetzt "+i+" Uhr.\n");
	call_out("WaitForFullHour",5);
}


// *** Funktionen fuer die Tagwechsel-Notifikations ***

static void StartWait()
{
  zone_t * zind;

  while (-1!=remove_call_out("WaitForChange"));
  
  zind = sort_array(m_indices(zones),
    lambda(({'a,'b}),({#'>,({#'[,zones,'a}),({#'[,zones,'b})})));

  call_out("WaitForChange",zones[zind[0],NEXT]-time());  
}

static void WaitForChange()
{
  zone_t * zind;
  
  zind = sort_array(m_indices(zones),
    lambda(({'a,'b}),({#'>,({#'[,zones,'a}),({#'[,zones,'b})})));

  // Next-Eintrag erneuern und STATE aendern    
  zones[zind[0],STATE] = (zones[zind[0],STATE]%4)+1;
	zones[zind[0],NEXT] = time() + zones[zind[0],zones[zind[0],STATE]]*60;
  MEMORY->put("zones", zones);
  
  // Grundsaetzlich wird ein Event verschickt
  send_event(ET_CHANGE_DAYTIME, ([E_DAYTIME:zones[zind[0],STATE],
                                  E_TIMEZONE:zind[0]]),
             ({}), EM_NO_CANCEL|EM_NO_MODIFY);

  // Bei Dusk und Dawn die Sequenzen laufen lassen  
  if (zones[zind[0],STATE]%2)
    ChangeOccurs(zind[0], 0, zones[zind[0],STATE]);

  StartWait();
}

// Wartet auf Lichtwechsel in einer Zeitzone
static void ChangeOccurs(zone_t zone, int seq, int state)
{
	map_objects(m_indices(mkmapping(map(users(),#'environment))),
	            "DuskDawn",zone,state,seq);
	if (seq!=6)
    call_out("ChangeOccurs", (zones[zone,state]*60)/6, zone, seq+1, state);  
}

// *** Funktionen zur Zeitzonen-Verwaltung ***

// Fuegt eine neue Zeitzone hinzu, damit man auch zur Laufzeit so etwas
// tun kann, ohne die Uhr neu laden zu muessen
public int AddTZ(zone_t tz, min_t off, min_t dusk, min_t day, min_t dawn,
                 min_t night, string desc, int flags) 
{
  time_t next;
  int state;
  
	if (!IS_ARCH(this_interactive())) 
	  return -1;
	if ( !intp(tz) || !intp(off) || !intp(dusk) || !intp(day) || !intp(dawn) 
	  || !intp(night) || !intp(flags))
	  return 0;

  next = get_0_oclock(time()) + off*60;
	zones += ([tz: 0; dusk; day; dawn; night; off; 0; 0; desc]);

  while (next < time())
  {
    state %= 4;
    state++;
    next += zones[tz,state]*60;
  }
  zones[tz, NEXT] = next;
  zones[tz, STATE] = state;

  printf("%O",zones);
  MEMORY->put("zones", zones);
  StartWait();

	return 1;
}

public int RemoveTZ(zone_t tz)
{
	if (!IS_ARCH(this_interactive())) 
	  return -1;
	if (!member(zones, tz) || tz == 0)
	  return 0;

  zones = m_delete(zones, tz);
  MEMORY->put("zones", zones);
  StartWait();

  return 1;  
}

public void PrintTZList()
{
  int * ind, i;
  
  printf(
    "+--------------------------------------------------------------------------+\n"
    "| Mud Clock Timezones                                                      |\n"
    "+--------------------------------------------------------------------------+\n"
    "| Zone             State  Next Change  Dusk   Day    Dawn   Night  Offset  |\n");
  for (i=0;i<sizeof(ind=m_indices(zones));i++)
  {
    printf(
      "| %:15-s  %:5-s  %:8s     %:5d  %:5d  %:5d  %:5d  %:5d   |\n",
      zones[ind[i],DESC],
      ({"Dusk","Day","Dawn","Night"})[zones[ind[i],STATE]-1],
      ctime(zones[ind[i],NEXT])[11..18],
      zones[ind[i],DUSK],
      zones[ind[i],DAY],
      zones[ind[i],DAWN],
      zones[ind[i],NIGHT],
      zones[ind[i],OFFSET]);
  }
  printf(
    "+--------------------------------------------------------------------------+\n");
}

// *** interne Hilfsfunktionen ***

// Liefert die Zeitzonennummer des aufrufenden Objektes oder von pl
// Der die Zeitzone bestimmende, umgebende Raum wird durchs Vorhandensein
// der Funktion int_long bestimmt.
private varargs zone_t get_tz(object pl) 
{
	object ob, env;
	zone_t i;
	
	if (!objectp(pl)) 
	  pl=0;
	ob=pl || previous_object();
	if (!ob) 
	  return 0;
	// den 'naechstliegendsten' Raum suchen
	while (!function_exists("int_long", ob)) 
	{
		env=environment(ob);
		if (!env) break;
		ob=env;
	}
	i=ob->QueryProp(P_TIMEZONE);
	if (!member(zones,i)) 
	  raise_error("undefinined timezone "+i);
	return i;
}

// Liefert Array mit ({ time_t next_day, time_t next_night })
private time_t* get_nexts(zone_t zone) 
{
	time_t length, i, j;
	
	if (!member(zones,zone)) 
	  return ({0, 0});
	
	switch (zones[zone,STATE])
	{
	  case DUSK:
	    return ({zones[zone,NEXT]+(zones[zone,DAY]+zones[zone,DAWN]+zones[zone,NIGHT])*60,
	             zones[zone,NEXT]+zones[zone,DAY]*60});
	    break;
	  case DAY:
	    return ({zones[zone,NEXT]+(zones[zone,DAWN]+zones[zone,NIGHT])*60,
	             zones[zone,NEXT]});
	    break;
	  case DAWN:
	    return ({zones[zone,NEXT]+zones[zone,NIGHT]*60,
	             zones[zone,NEXT]+(zones[zone,NIGHT]+zones[zone,DUSK]+zones[zone,DAY])*60});
	    break;
	  case NIGHT:  
	    return ({zones[zone,NEXT],
	             zones[zone,NEXT]+(zones[zone,DUSK]+zones[zone,DAY])*60});
	    break;
	}
}

// Meldung an alle hoerende Spieler ausgeben
private void shoutout(string txt) {
	object* pls;
	int i;
	pls=users();
	for (i=sizeof(pls); i--; ) {
		if (pls[i]->CannotHear(1)) continue;
		tell_object(pls[i], txt);
	}
}

// *** Hilfsfunktionen fuer RL-Zeit Berechnungen ***

// Liefert die zum Zeitpunkt 'ticks' an demselben Tag schon
// verstrichenen Sekunden
// Nuetzlich um danach mit den Sekundenzahlen Rechenoperationen
// durchzufuehren (Notwendig, da '0 ticks' 1 Uhr bzw 2 Uhr ist...)
// 0.00:00 Uhr -> 0, 23.59:59 -> 86399
public int get_day_ticks(time_t ticks) {
	int h, min, sec;
	sscanf(ctime(ticks), "%!s %!s %!d %d:%d:%d %!d", h, min, sec);
	return 3600*h+60*min+sec;
}

// Liefert 0:00 Uhr des durch 'ticks' bestimmten Tages
// zB: ctime(get_0_oclock(time())) -> Heute, 0 Uhr
public time_t get_0_oclock(time_t ticks) {
	return ticks-get_day_ticks(ticks);
}


// *** Funktionen fuers Debugging ***

mixed* _zones(mixed* x) {
	if (!IS_ARCH(this_interactive())) return copy(zones);
	if (x) MEMORY->put("zones", x);
	return zones=MEMORY->get("zones");
}
