/*******************
** Eldarea MUDLib **
********************
**
** secure/eventd.c - event daemon
**
** CVS DATA
** $Date: 2000/12/12 12:28:08 $
** $Revision: 1.2 $
**
** CVS History
**
** $Log: eventd.c,v $
** Revision 1.2  2000/12/12 12:28:08  elatar
** new eventd from WL
**
**
*/

#pragma strong_types
#pragma no_inherit
#pragma no_clone
#pragma no_shadow

#include <events.h>
#include <config.h>

//#define DEBUG       1
//#define DEBUG2      1
//#define STATISTIK   1

// Um zu tiefe Rekursion zu verhindern
#define MAX_STACK 30

#define PRELOADFILE    "/etc/eventd.handlers"
#define SECUREFILE     "/etc/eventd.secure"
#define TRUSTFILE      "/etc/eventd.trust"
#define GLOBTRUSTFILE  "/etc/eventd.globtrust"
#define LOGFILE        "events/DAEMON"
#define CORE_LOGFILE   "events/DAEMON_CORE"
#define GLOBAL_LOGFILE "events/GLOBAL_LISTEN"
#define LISTEN_LOGFILE "events/LISTEN"
#define LDATE          ctime()[4..18]

private static mapping events;
// Key ist Event-Typ. Value ist wieder ein Mapping
// dessen Key ein Objekt ist, Value 0 ist ein Array
// aus Prioritaeten und Value 1 ist ein Array aus Closures.
// Value 0 und Value 1 werden zusammen als alist behandelt.
// ([ object : ({ prio }); ({ callback }) ])

private static mapping default_handlers;
// Key ist Event-Typ. Value ist Filename des Default-Handlers

private static mapping global;
// Key ist Event-Typ. Value ist Array aus Objekten

private static mapping security;
// Key ist Event-Typ.
// Value 0 ist Integer (bitweises OR)
// - 1 globale Sicherheit
// - 2 volle Sicherheit
// - 4 globale Listener mitloggen
// - 8 Listener mitloggen
// Value 1 ist Array aus Objektnamen global getrusted
// Value 2 ist Array aus Objektnamen voll getrusted

// temporaere Variable. Der Inhalt ist nur innerhalb eines
// EINZELNEN Event-Cycles aktuell. Bitte mit AUESSERSTER VORSICHT
// verwenden! Jede Rekursion setzt diese Variable neu!
private static mixed tmp_event_referenz;

#ifdef STATISTIK
// Statistik-Mapping. Key ist Event-Typ.
// Value 0 ist Anzahl der aufgerufenen Events dieses Typs
private static mapping stats;
#endif

// Prototypen
private mapping preload_handlers();
private mapping preload_security();
private void clean_me();

void create() {
  mixed info, err, res;
  seteuid(getuid(this_object()));
  if(mappingp(events))
    return;

  default_handlers = preload_handlers();
  security = preload_security();

  // Gibt es schon Eintraege im Dauerspeicher?
  err=catch(res=MEMORY->test("events"));

  // Bei Fehler Notbetrieb ermoeglichen
  if(err) {
    events = ([]); global = ([]);
    raise_error(MEMORY" "+err);
  }

  // Initialisieren, wenn der Dauerspeicher keine Daten enthaelt
  if(!res) {
    MEMORY->put( "events", ([]) );
    MEMORY->put( "global", ([]) );
  }

  // Referenzieren
  events = MEMORY->get( "events" );
  global = MEMORY->get( "global" );

#ifdef STATISTIK
  stats=([]);
#endif

  tmp_event_referenz = ({});

}

private int allowed(object ob, string type, int prio, closure callback) {
  int sec;
  string fil;
  // Closures in fremden Objekten verboten
  if(ob!=to_object(callback))
    return 0;
  // Default-Handler duerfen immer
  if(type && default_handlers[type]==load_name(ob))
    return 1;
  // Sicherheit gesetzt?
  if(sec=security[type, 0]) {
    if(sec&2) {
      fil=load_name(ob);
      // Lauscher muss getrustet sein (voll oder global)
      if(member_array(fil, security[type, 2])<0 
        && member_array(fil, security[type, 1])<0) {
        log_file(LISTEN_LOGFILE, LDATE+": "+object_name(ob)+
          " wants ILLEGAL listen '"+upper_case(type)+"'\n");
        raise_error(object_name(ob)+" darf nicht "+
          upper_case(type)+" lauschen!\n");
      }
    }
    if(sec&8) {
      log_file(LISTEN_LOGFILE, LDATE+": "+object_name(ob)+
        " listening '"+upper_case(type)+"'\n");
    }
  }
  return 1;
}

private int global_allowed(object ob, string type) {
  int sec;
  // Default-Handler duerfen immer
  if(ob && type && default_handlers[type]==load_name(ob))
    return 1;
  // Sicherheit gesetzt?
  if(sec=security[type, 0]) {
    if(sec&1 || sec&2) {
      if(member_array(load_name(ob), security[type, 1])<0) {
        log_file(GLOBAL_LOGFILE, LDATE+": "+object_name(ob)+
          " wants ILLEGAL listen '"+upper_case(type)+"'\n");
        raise_error(object_name(ob)+" darf nicht global "+
          upper_case(type)+" lauschen!\n");
      }
    }
    if(sec&4) {
      log_file(GLOBAL_LOGFILE, LDATE+": "+object_name(ob)+
        " listening '"+upper_case(type)+"'\n");
    }
  }
  return 1;
}

// Preloading der Default Handler
private mapping preload_handlers() {
  string f;
  int i;
  mixed lines, tmp;
  mapping ret;
  ret = ([]);
  if(!stringp(f=read_file(PRELOADFILE) )) {
    log_file(LOGFILE,
      LDATE+": Kein preloading.\n");
    return ret;
  }
  lines=efun::explode(f,"\n")-({""});
  for(i=0;i<sizeof(lines);i++) {
    tmp = efun::explode(lines[i]," ")-({""});
    if(!sizeof(tmp) || tmp[0][0]=='#') continue;
    if(sizeof(tmp)!=2) {
      log_file(LOGFILE, LDATE+
        ": Fehler im Preload-File ("+(string)(i+1)+").\n");
      continue;
    }
    if(file_size(tmp[1]+".c")<0) {
      log_file(LOGFILE, LDATE+
        ": Handler existiert nicht im Preload-File ("+(string)(i+1)+").\n");
      continue;
    }
    ret+=([tmp[0]:tmp[1]]);
  }
  log_file(LOGFILE, sprintf("%s: %d Handler geladen.\n  %s\n",
    LDATE,
    sizeof(ret),
    implode(map_array(m_indices(ret), #'upper_case), ", ")));
  return ret;
}

// Preloading der Event-Sicherheit
private mapping preload_security() {
  string f;
  int i;
  mixed lines, tmp;
  mapping ret;
  ret = ([]);
  // zuerst Secure-File lesen
  if(!stringp(f=read_file(SECUREFILE) )) {
    log_file(LOGFILE,
      LDATE+": Keine Sicherheit.\n");
    return ret;
  }
  lines=efun::explode(f,"\n")-({""});
  for(i=0;i<sizeof(lines);i++) {
    tmp = efun::explode(lines[i]," ")-({""});
    if(!sizeof(tmp) || tmp[0][0]=='#') continue;
    if(sizeof(tmp)!=2) {
      log_file(LOGFILE, LDATE+
        ": Fehler im Security-File ("+(string)(i+1)+").\n");
      continue;
    }
    ret+=([tmp[0]:to_int(tmp[1]);({});({})]);
  }
  // Globtrust-File lesen
  if(stringp(f=read_file(GLOBTRUSTFILE) )) {
    lines=efun::explode(f,"\n")-({""});
    for(i=0;i<sizeof(lines);i++) {
      tmp = efun::explode(lines[i]," ")-({""});
      if(!sizeof(tmp) || tmp[0][0]=='#') continue;
      if(sizeof(tmp)<2) {
        log_file(LOGFILE, LDATE+
          ": Fehler im Globtrust-File ("+(string)(i+1)+").\n");
        continue;
      }
      if(!member(ret, tmp[0]) || !(ret[tmp[0], 0]&1))
        continue;
      ret[tmp[0], 1]+=tmp[1..];
    }
  }
  // Trust-File lesen
  if(stringp(f=read_file(TRUSTFILE) )) {
    lines=efun::explode(f,"\n")-({""});
    for(i=0;i<sizeof(lines);i++) {
      tmp = efun::explode(lines[i]," ")-({""});
      if(!sizeof(tmp) || tmp[0][0]=='#') continue;
      if(sizeof(tmp)<2) {
        log_file(LOGFILE, LDATE+
          ": Fehler im Trust-File ("+(string)(i+1)+").\n");
        continue;
      }
      if(!member(ret, tmp[0]) || !(ret[tmp[0], 0]&2))
        continue;
      ret[tmp[0], 2]+=tmp[1..];
    }
  }
  return ret;
}

varargs mixed query_default_handlers(string type) {
  if(stringp(type) && strlen(type))
    return deep_copy(default_handlers[type]);
  return deep_copy(default_handlers);
}

int listen(string type, int prio, mixed callback) {
  mapping list;
  int *prios, i;
  closure *callbacks;
  mixed temp;
  object po;

#ifdef DEBUG
  printf("LISTEN Type: %s\nCALLBACK: %O\n",type,callback);
#endif

  if (!stringp(type) || !intp(prio) || !closurep(callback))
    return 0;
  // -1 obsolete. ungueltige Typen gibts derzeit nicht
  if(!allowed((po=previous_object()), type, prio, callback))
    return -2;
  // diesen Event gips noch nich
  if (!mapping_contains(&list, events, type)) {
    events[type]  = ([ po : ({ prio }); ({ callback }) ]);
    return 1;
  }
  // dieses Objekt lauscht noch nich
  if (!mapping_contains(&prios, &callbacks, list, po)) {
    list += ([ po : ({ prio }); ({ callback }) ]);
    return 1;
  }
  // Verhinderung identischer Eintraege
  temp = sprintf("%O", callback);
  for (i=sizeof(callbacks); i--;) {
    if (sprintf("%O", callbacks[i])==temp && prios[i]==prio) return 1;
  }
  // Fuege ein weiteres hinzu
  temp = order_alist(prios + ({ prio }), callbacks + ({ callback }) );
  list += ([ po : temp[0]; temp[1] ]);
  return 1;
}


// Liefert alle Lauscher oder alle Lauscher auf einen bestimmten Typ
public varargs mixed query_listeners(string type)
{
	if(stringp(type) && strlen(type))
		return deep_copy(events[type]);
	return deep_copy(events);
}


// Liefert fuer 'ob' ein mapping: ([ string typ: mixed alist ])
// 'alist' sieht wie folgt aus: ({ int *prios, closure *cls})
// Lauscht 'ob' keinem Event, wird 0 returnt
public mapping is_listening(object ob)
{
	mixed tmp;
	if(!objectp(ob)) return 0;
	// In tmp ein mapping erstellen
	tmp=map(events, lambda( ({'et}),
		({#'?,
			({#'member, ({#'[, events, 'et}), ob}),
			({#'({,
					({#'[, ({#'[, events, 'et}), ob, 0}), 
					({#'[, ({#'[, events, 'et}), ob, 1}) 
			})
		}) ));
	// alle raus mit m_value == 0
	tmp=filter(tmp, lambda( ({'x}), ({#'[, tmp, 'x}) ) );
	if(!sizeof(tmp)) return 0;
	return deep_copy(tmp);
}


void unlisten(string type, int prio, mixed callback) {
  mapping list;
  int *prios;
  closure *callbacks;
  int pos;

  if (!(list = events[type]))
    return;
  if (!mapping_contains(&prios,&callbacks,list,previous_object()))
    return;
  if ((pos = member_array(prio,prios)) < 0)
    return;
  // Schmeisst erste Uebereinstimmung von prio und callback raus
  while (pos < sizeof(prios) && prios[pos]==prio) {
#if 0  // Vergleichen von lfun-closures geht nicht :-(
    if (callbacks[pos]==callback) {
#else
    if (sprintf("%O",callbacks[pos])==sprintf("%O",callback)) {
#endif
      prios[pos..pos]=({ });
      callbacks[pos..pos]=({ });
      if (!sizeof(prios))
        m_delete(list,previous_object());
      else
        list += ([ previous_object() : prios; callbacks ]);
      return;
    }
    else
      pos++;
  }
}


public void set_global(string type, status state)
{
	object po;
	if(!global_allowed((po=previous_object()), type))
		return;
	// global-listener setzen
	if(state)
	{
		if(!global[type] || member(global[type],po) < 0)
			global[type] = (global[type]||({}))-({0}) + ({po});
		return;
	}
	// Ansonsten global-listener entfernen
	if(global[type]) global[type] -= ({po,0});
	return;
}


public varargs mixed query_global(string type)
{
	if(stringp(type) && strlen(type))
		return deep_copy(global[type]);
	return deep_copy(global);
}


// laufenden Event als abgebrochen markieren. Laeuft grad kein Event,
// oder ist die aktuelle Prio < 0 oder ist der Event nicht abbrechbar,
// dann return 0, ansonsten als abgebrochen kennzeichnen
public int cancel_event(mixed info)
{
	if (!sizeof(tmp_event_referenz) || tmp_event_referenz[<1][3][0][<1]<0 ||
		tmp_event_referenz[<1][2] & EM_NO_CANCEL) return 0;
	tmp_event_referenz[<1][1][E_CANCELLED] = info || 1;
	tmp_event_referenz[<1][1][E_CANCELLER] = previous_object();
	return 1;
}


// laufenden Event als gehandelt markieren
public int handle_event(mixed info)
{
	if (!sizeof(tmp_event_referenz)) return 0;
	tmp_event_referenz[<1][1][E_HANDLED] = info || 1;
	tmp_event_referenz[<1][1][E_HANDLER] = previous_object();
	return 1;
}


// Arbeitet den Event 'ev' ab. Die Funktion kann sich selbst rekursiv
// aufrufen (ein event loest einen anderen aus), daher sind hier globale
// Variablen ziemlich 'kritisch'
private mixed process(mixed ev) {
  mixed mode;

  // Hauptschleife solange noch calls da sind
  while(sizeof(ev[3][0])) {

    // Temp. Variable setzen fuer cancel_event() und handle_event()
    // Ueberlauf vorbeugen durch Bugs etc.
    tmp_event_referenz += ({ ev });
    if (sizeof(tmp_event_referenz)>MAX_STACK)
        tmp_event_referenz=tmp_event_referenz[<MAX_STACK..];

    if(to_object(ev[3][1][<1])) {

      // das if() hier ist nur Optimierung des EM_SIMPLE Mode.
      // nur wenn 'mode' gesetzt ist, schau nach, was...
      if(mode=ev[2]) {

        // Im Debug-Mode Startkosten setzen
        if( mode & EM_DEBUG ) 
          ev[1][E_DEBUG_INFO][2][sizeof(ev[3][0])-1] =
            get_eval_cost();

        // einfache Kopie der Daten
        if( mode & EM_COPY )
          funcall(ev[3][1][<1],
            copy(ev[1]),
            ev[0],
            ev[3][0][<1],
            mode);

        // echte Kopie der Daten (keine Referenzen mehr)
	// macht auch das EM_NO_MODIFY (alias)
        else if( mode & EM_DEEP_COPY )
          funcall(ev[3][1][<1],
            deep_copy(ev[1]),
            ev[0],
            ev[3][0][<1],
            mode);

        // wir tun nur so als ob.
        else if( mode & EM_FAKE ); // do nothing

        // ansonsten einfach ohne Referenzen
        else 
          funcall(ev[3][1][<1], 
            ev[1], 
            ev[0], 
            ev[3][0][<1], 
            mode);
      }

      // Standardfall ohne Mode ( ->EM_SIMPLE )
      else
        funcall(ev[3][1][<1],
          ev[1],
          ev[0],
          ev[3][0][<1],
          mode);

    }

    // Im Debug-Mode Eval-Kosten ausrechnen
    if( ev[2]&EM_DEBUG ) 
      ev[1][E_DEBUG_INFO][2][sizeof(ev[3][0])-1] -=
        get_eval_cost();

    // Event wird abgebrochen?
    if (ev[1][E_CANCELLED])
    {
	// wenn Prio < 0 -> abbrechen verboten
        if(ev[3][0][<1]<0)
	{
		ev[1][E_CANCELLED]=0;
		ev[1][E_CANCELLER]=0;
	}
	else break;
    }

    // Temporaere Variable wieder loeschen
    tmp_event_referenz = tmp_event_referenz[0..<2];

    // abgearbeiteten callback aus der Liste nehmen
    ev[3][0]=ev[3][0][0..<2];
    ev[3][1]=ev[3][1][0..<2];

  } // while

  // Data-Mapping zurueckgeben
  return ev[1];
}

mixed send(string type, mapping data, mixed dest, mixed mode) {
  int i;
  mapping ev_filter;
  object *ev_obs;
  mixed ev_prios, ev_calls, prios, callbacks;
  string vb;

  if(!stringp(type) || !mappingp(data))
    return 0;

  if(caller_stack_depth()>=MAX_STACK) {
    log_file(CORE_LOGFILE, LDATE+" MAX_STACK ueberschritten.\n"+
      sprintf("TYP : %s\n",upper_case(type))+
      sprintf("DATA :\n%O\n",data)+
      sprintf("THIS_PLAYER : %O\n",this_player())+
      sprintf("THIS_INTERACTIVE : %O\n",this_interactive())+
      sprintf("CALLER-STACK :\n%O\n\n", caller_stack()));
    log_file(LOGFILE,sprintf("%s: Zu tiefe Rekursion.\n  Ausloeser: %O\n",
      LDATE, (caller_stack()||({0}))[0]));
    raise_error("MAX_STACK ueberschritten. /log/"CORE_LOGFILE
      " erzeugt.\n");
  }

  dest = filter_array( (pointerp(dest)?dest:({dest})), #'objectp);

  data[E_SENDER]       = previous_object();
  data[E_DESTINATIONS] = dest;

  if(stringp(vb=query_command()))
    data[E_COMMAND] = vb;

#ifdef DEBUG
  printf("SEND TYPE: %s\nSEND MODE: %d\nDESTINATIONS:\n%s\n",
    type,
mode,
    implode(map_array(dest,#'file_name),"\n"));
#endif

  // Default-Handler laden
  // Diese muessen sich selbst als globale Lauscher anmelden
  if(member(default_handlers,type))
    load_object(default_handlers[type]);

  // Gibts Lauscher fuer diesen Event? 
  if(!(ev_filter = events[type])) 
    return data; 

  // Keine konkreten Empfaenger? Dann an alle
  if(!sizeof(dest))
    ev_obs = m_indices(ev_filter);
  else {
    // Sammelt alle Empfaenger
    // Globalen Lauscher immer dazu (vorher Nullpointer raus)
    if(pointerp(global[type])) dest+=(global[type]-=({0}));
    for( ev_obs=({}),i=sizeof(dest); i--; )
      ev_obs += filter_array(deep_inventory(dest[i])+dest[i..i], ev_filter);
    // Doppelte rauskicken
    ev_obs=m_indices(mkmapping(ev_obs));
  }

  // Im Complex-Mode E_RECEIVERS setzen
  if( mode & EM_COMPLEX )
    data[E_RECEIVERS] = ev_obs;

#ifdef DEBUG
  printf("RECEIVERS:\n%s\n",
    implode(map_array(ev_obs,#'file_name),"\n"));
#endif

  // Prios & Callbacks der Objekte sammeln
  for(ev_prios = ({}),ev_calls = ({}),i=sizeof(ev_obs);i--;) {
    mapping_contains( &prios, &callbacks, ev_filter, ev_obs[i] );
    ev_prios += ( pointerp(prios) ? prios: ({prios}) );
    ev_calls += ( pointerp(callbacks) ? callbacks: ({callbacks}) );
  }

  // Prios sortieren
  ev_prios = order_alist( ev_prios, ev_calls );

#ifdef DEBUG
  printf( "SORTED: %O\n", ev_prios );
#endif

  // Im Debug-Mode E_DEBUG_INFO fuellen
  if( mode & EM_DEBUG )
    data[E_DEBUG_INFO] = ({}) + ev_prios + ({ allocate(sizeof(ev_prios[0])) });

#ifdef STATISTIK
  if(!member(stats, type)) stats+=([type:0]);
  stats[type, 0]++;
#endif

  // Den Event abarbeiten.
  return process( ({ type, data, mode, ev_prios }) );
}

public string statistik() {
#ifdef STATISTIK
  string txt;
  int i;
  mixed known;
  known=m_indices(stats);
  txt ="Event-Daemon Statistik vom "+LDATE+"\n";
  txt+="------------------------------------------\n";
  txt+="Benutzte Event-Typen ("+to_string(sizeof(known))+"):\n";
  txt+=implode(map_array(known, #'upper_case), ", ");
  txt+="\n\n";
  known+=m_indices(default_handlers);
  known+=m_indices(global);
  known+=m_indices(events);
  known+=m_indices(security);
  known=sort_array(m_indices(mkmapping(known)), #'<);
  txt+="Event-Typ        | Call | Secu | Default-Handler\n";
  txt+="-----------------+------+------+----------------\n";
  for(i=sizeof(known);i--;) {
    txt+=sprintf("%-16s | %:4d | %:4d | %-s\n",
      upper_case(known[i]),
      stats[known[i], 0],
      security[known[i], 0],
      (string)default_handlers[known[i]]);
  }
  return break_string(txt);
#else
  return "Keine Statistik verfuegbar!\n";
#endif
}


// Raeumt bissl auf, damit sich nicht so viel anhaeuft...
private void clean_me()
{
	mixed idx;
	int i;

	// leere Eintraege entfernen im events-mapping
	idx=m_indices(events);
	for(i=sizeof(idx);i--;)
		if(!mappingp(events[idx[i]]) || !sizeof(events[idx[i]]))
			m_delete(events, idx[i]);
}


void reset()
{
	clean_me();
}


