/*******************
** Eldarea MUDLib **
********************
**
** secure/inetd.c - udp port handling
**
** CVS DATA
** $Date: 2001/08/21 11:53:49 $
** $Revision: 1.3 $
**
** UDP port handling code. Version 0.7a
** Written by Nostradamus for Zebedee.
** Developed from an original concept by Alvin@Sushi.
**
** CVS History
**
** $Log: inetd.c,v $
** Revision 1.3  2001/08/21 11:53:49  eldarea
** corrected send_udp() prototype
**
** Revision 1.2  2000/11/30 16:07:32  elatar
** header corrected
**
** Revision 1.1.1.1  1999/11/05 12:30:45  elatar
** Preparing mudlib for cvs control
**
**
*/

#pragma strict_types
#include <udp.h>
#include <wizlevels.h>

/* Public commands which will be accessible to any unlisted muds.
 * PING, QUERY and REPLY are included by default. */
#define COMMANDS \
({ "channel", "finger", "tell", "who", "mail", "www", "htmlwho",\
   "man" })

/* Define this to the object that receives incoming packets and passes
 * them to the inetd. Undefine for no receive_udp() security.
 * NOTE: The string must be of the format that file_name() returns. */
#define UDP_MASTER		__MASTER_OBJECT__

/* The maximum number of characters we can send in one packet. */
#define MAX_PACKET_LEN	1024

#ifndef DATE
#define DATE		ctime(time())[4..15]
#endif

#define UP		time()
#define DOWN		(-time())

#define DELIMITER	"|"
#define HOSTFILE_DEL	":"
#define HOSTFILE_DEL2	","
#define RETRY		"_RETRY"
#define SAVEFILE        "/etc/inetd"

/*
 * after 3 months without connect to a mud, delete it
 * from INETD_HOSTS list. We guess its dead...
 */
#define EXPIRE      30*24*60*60    // 30 Tage
#define PING_DELAY  12             // alle 12 Resets ping alle Muds an

mapping hosts;
private static mapping pending_data, incoming_packets;
private static string *received_ids;
private static int packet_id, counter;
private static object debug; /* wenn != 0 kein catch() */
private mapping statistics;

/* for manual re-reading of the INETD_HOSTS list or via /room/mkz */
void read_host_list();
void dump_host_list();

varargs string send_udp(string mudname, mapping data, int expect_reply);

void create() {
  seteuid(getuid());
  packet_id = 0;
  pending_data = ([ ]);
  incoming_packets = ([ ]);
  received_ids = ({ });
  if(!restore_object(SAVEFILE) || !mappingp(hosts)) {
    hosts = ([ ]);
  }
  if(!mappingp(statistics)) statistics = ([ "StartStatistics": ctime() ]);
  if(!this_player())
    call_out("reset", 60*5, 0); // after 5 minutes force reset
}

status check_system_field(mapping data, string field) {
    return data[SYSTEM] && member_array(field, data[SYSTEM]) != -1;
}

void __add_host(mapping host)
{
  if (!IS_ARCH(this_interactive())) return;

  hosts += ([host]);
  dump_host_list();
}

mapping add_system_field(mapping data, string field) {
  if(data[SYSTEM]) {
    if(!check_system_field(data, field))
      data[SYSTEM] += ({ field });
  }
  else
    data[SYSTEM] = ({ field });
  return data;
}

/*
 * Read the INETD_HOSTS file and set the "hosts" mapping from it.
 * Retain existing HOST_STATUS fields.
 */
void read_host_list() {
  mixed data;

  if(data = read_file(HOST_FILE)) {
    mapping old_hosts;
    int i, stat;
    string *local_cmds;
    string name;

    old_hosts = copy_mapping(hosts);
    hosts = ([]);
    for(i = sizeof(data = explode(data, "\n")); i--; ) {
      if(data[i] == "" || data[i][0] == '#')
        continue;
      if(sizeof(data[i] = explode(data[i], HOSTFILE_DEL)) < 5) {
        log_file(INETD_LOG_FILE, DATE+": error in '"+HOST_FILE+
          "' line "+(i + 1)+"\n");
        continue;
      }
      name = lower_case(data[i][HOST_NAME]);
      if(member_array("*",
        local_cmds = explode(data[i][LOCAL_COMMANDS], HOSTFILE_DEL2)) != -1)
          local_cmds = local_cmds - ({ "*" }) + COMMANDS;
      hosts[name] = ({
        capitalize(data[i][HOST_NAME]),
        data[i][HOST_IP],
        to_int(data[i][HOST_UDP_PORT]),
        local_cmds,
        explode(data[i][HOST_COMMANDS], HOSTFILE_DEL2),
        DOWN});
      /*
       * Retain existing host status as long as the IP and
       * UDP ports are the same.
       */
      if(old_hosts[name] &&
        old_hosts[name][HOST_IP] == hosts[name][HOST_IP] &&
        old_hosts[name][HOST_UDP_PORT] == hosts[name][HOST_UDP_PORT])
          hosts[name][HOST_STATUS] = old_hosts[name][HOST_STATUS];
    }
    save_object(SAVEFILE);
    log_file(INETD_LOG_FILE,
      DATE+": read '"+HOST_FILE+"' sucessfully.\n");
  }
  else
    log_file(INETD_LOG_FILE, 
      DATE+": error reading '"+HOST_FILE+"'.\n");
}

/*
 * dump of whole internal list to HOST_FILE
 */
varargs void dump_host_list() {
  int i,j;
  mixed *keys;
  mixed *tmp,tmp2;

  rename(HOST_FILE, HOST_FILE+".old");
  keys=m_indices(hosts);
  for(i=0;i<sizeof(keys); i++) {
    if(!pointerp(tmp=hosts[keys[i]]) || sizeof(tmp)<5) continue;
    write_file(HOST_FILE,
      sprintf("%s:%s:%d:%s:%s\n",tmp[0],tmp[1],tmp[2],"*",implode(tmp[4],",")));
  }
  log_file(INETD_LOG_FILE, 
    DATE+": dumped '"+HOST_FILE+"' sucessfully.\n");
}

void check_mud(string mudname) {
  int last, i;
  mixed muds;
  for(i=sizeof(muds=m_indices(hosts));i--;) {
    last=hosts[muds[i]][HOST_STATUS];
    if(last<0) {
      if(abs(last)+EXPIRE<time()) {
        hosts=m_delete(hosts, muds[i]);
        log_file(INETD_LOG_FILE,
          DATE+": mud expired from list: "+muds[i]+".\n");
      }
    }
  }
  save_object(SAVEFILE);
}

/*
 * Make a PING request to all muds in the "hosts" mapping to set
 * HOST_STATUS information.
 */
void startup() {
  mixed muds;
  int i;
  for(i=sizeof(muds=m_indices(hosts));i--;) {
    hosts[muds[i]][HOST_STATUS]=-abs(hosts[muds[i]][HOST_STATUS]); // set DOWN
    send_udp(muds[i], ([ REQUEST: PING ]), 1);                     // and PING
  }
  call_out("check_mud",240);
}

/*
 * Remove a buffered packet from the "incoming_packets" mapping.
 */
void remove_incoming(string id) {
    incoming_packets = m_delete(incoming_packets, id);
}

/*
 * Decode a string from a UDP packet.
 * Returns:   The actual value of the argument (either int or string)
 */
mixed decode(string arg) {
  if(arg[0] == '$')
    return arg[1..];
  if((string)((int)arg) == arg)
    return (int)arg;
  return arg;
}

/*
 * Decode a UDP packet.
 * Arguments: UDP packet as a string.
 * Returns:   The decoded information as a mapping, 1 for succes but no
 *            output (buffered packet), or 0 on error.
 */
mixed decode_packet(string packet, string delimiter) {
    string *data;
    mapping ret;
    string info, tmp;
    mixed class;
    int i, id, n;

    /* If this packet has been split, handle buffering. */
    if (packet[0..strlen(PACKET)] == PACKET + ":") {
	if (sscanf(packet, PACKET + ":%s:%d:%d/%d" + delimiter + "%s",
	class, id, i, n, tmp) != 5)
	    return 0;
	class = lower_case(class) + ":" + id;
	if (pointerp(incoming_packets[class])) {
	    incoming_packets[class][i-1] = tmp;
	    if (member_array(0, incoming_packets[class]) == -1) {
		ret =
		decode_packet(implode(incoming_packets[class], ""), delimiter);
		remove_incoming(class);
		return ret;
	    }
	} else {
	    incoming_packets[class] = allocate(n);
	    incoming_packets[class][i-1] = tmp;
            /* Is it possible to already have a timeout running here ?!? */
	    /* If no timeout is running then start one. */
	    if (!pending_data[class]) {
		call_out("remove_incoming",
		REPLY_TIME_OUT + REPLY_TIME_OUT * RETRIES, class);
	    } else {
		//DEBUG("\nINETD: *** Buffered packet Timeout already running! ***\n");
	    }
	}
	return 1;
    }
    ret = ([ ]);
    for(i = 0, n = sizeof(data = explode(packet, delimiter)); i < n; i++) {
	/* DATA fields can be denoted by a preceeding blank field. */
	if (data[i] == "") {
	    tmp = DATA;
	    /* Test for illegal packet length (no DATA) */
	    if (++i >= n)
		return 0;
	    info = data[i];
	}
	else if (sscanf(data[i], "%s:%s", tmp, info) != 2)
	    return 0;
	switch((string)(class = decode(tmp))) {
	    case DATA:
		return ret + ([ DATA: decode(implode(
		    ({ info }) + data[i+1..], delimiter)) ]);
	    case SYSTEM:
		ret[class] = explode(info, ":");
		continue;
	    default:
		ret[class] = decode(info);
		continue;
	}
    }
    return ret;
}

/*
 * Check wether a UDP packet was valid.
 * Logs are made and "host" information is updated as appropriate.
 * Arguments: Decoded UDP packet (mapping)
 * Returns:   0 for valid packets, an error string otherwise.
 */
string valid_request(mapping data) {
  mixed host_data;
  string *muds;
  string req;
  int i;

  if(!data[NAME] || !data[UDP_PORT])
    return DATE + ": Illegal packet.\n";
  if (data[NAME] && stringp(data[REQUEST]) && data[REQUEST]!="reply") {
    if (!member(statistics, data[NAME]))
      statistics+=([ data[NAME]: ([ data[REQUEST]: 1 ]) ]);
    else {
      if (!member(statistics[data[NAME]], data[REQUEST]))
        statistics[data[NAME]]+=([ data[REQUEST]: 1 ]);
      else statistics[data[NAME]][data[REQUEST]]++;
    }
  }
  if(host_data = hosts[lower_case(data[NAME])]) {
    if(data[HOST] != host_data[HOST_IP]) {
      if(data[NAME] == LOCAL_NAME)
        return DATE + ": *** FAKE MUD ***\n";
      log_file(INETD_LOG_FILE,
        DATE+": host change: "+host_data[HOST_NAME]+": "+
        host_data[HOST_IP]+" -> "+data[HOST]+"\n");
      host_data[HOST_IP] = data[HOST];
      save_object(SAVEFILE);
    }
    if(data[UDP_PORT] != host_data[HOST_UDP_PORT]) {
      if(data[NAME] == LOCAL_NAME)
        return DATE + ": *** FAKE MUD ***\n";
      log_file(INETD_LOG_FILE,
        DATE+": port change: "+host_data[HOST_NAME]+" "+
        host_data[HOST_UDP_PORT] + " -> " + data[UDP_PORT] + "\n");
      host_data[HOST_UDP_PORT] = data[UDP_PORT];
      save_object(SAVEFILE);
    }
  } 
  else {
    if(lower_case(data[NAME]) == lower_case(LOCAL_NAME))
      return DATE + ": *** FAKE MUD ***\n";
    for(i = sizeof(muds = m_indices(hosts)); i--; ) {
      host_data = hosts[muds[i]];
      if(data[HOST] == host_data[HOST_IP] &&
          data[UDP_PORT] == host_data[HOST_UDP_PORT]) {
        log_file(INETD_LOG_FILE,
          DATE+": name change: "+host_data[HOST_NAME]+" -> "+
          data[NAME] + "\n");
        host_data[HOST_NAME] = data[NAME];
        hosts[lower_case(data[NAME])] = host_data;
        hosts = m_delete(hosts, muds[i]);
        save_object(SAVEFILE);
        i = -2;
        break;
      }
    }
    if(i != -2) {
      host_data = hosts[lower_case(data[NAME])] = ({
      data[NAME],
      data[HOST],
      data[UDP_PORT],
      COMMANDS,
      ({ "*" }),
      UP });
      log_file(INETD_LOG_FILE, 
        DATE+": new mud: "+data[NAME]+":"+data[HOST]+":"+data[UDP_PORT]+"\n");
      save_object(SAVEFILE);
    }
  }
  if(!(req = data[REQUEST]))
    return DATE + ": System message.\n";
  if(req != PING &&
      req != QUERY &&
      req != REPLY &&
      member_array(req, host_data[LOCAL_COMMANDS]) == -1) {
    /* This should probably send a system message too. */
    send_udp(host_data[HOST_NAME], ([
    REQUEST: REPLY,
    RECIPIENT: data[SENDER],
    ID: data[ID],
    DATA: "Invalid request @" + LOCAL_NAME + ": " + 
    capitalize(data[REQUEST]) + "\n" ]) );
    return DATE + ": Invalid request.\n";
  }
  return 0;
}

/*
 * Incoming UDP packets are sent to this function to be interpretted.
 * The packet is decoded, checked for validity, HOST_STATUS is updated
 * and the appropriate udp module called.
 * Arguments: Senders IP address (string)
 *            UDP packet (string)
 */
void receive_udp(string sender, string packet) {
    mapping data;
    string req, err, id;

#ifdef UDP_MASTER
    if (!previous_object() ||
    file_name(previous_object()) != UDP_MASTER) {
	log_file(INETD_LOG_FILE, DATE+": illegal: receive_udp() by " +
	file_name(previous_object()) + "\n");
	return;
    }
#endif

    if (
	!(data = decode_packet(packet, DELIMITER))
    ) {
	if (!data)
	    log_file(INETD_LOG_FILE, DATE + ": Received invalid packet.\n"
              "  Sender: "+sender+"\n  Packet:\n  " + packet + "\n");
	return;
    }
	  if (!mappingp(data))
		  return;
    data[HOST] = sender;
    if (err = valid_request(data))
    {
        if (data[REQUEST]!="locate")
	    log_file(INETD_LOG_FILE, err + "\n  Sender: " + sender + 
                "\n  Packet:\n  "+packet + "\n");
      return;
    }
    hosts[lower_case(data[NAME])][HOST_STATUS] = UP;
    if ((req = data[REQUEST]) == REPLY) {
    	mapping pending;

	/* If we can't find the reply in the pending list then bin it. */
	if (!(pending = pending_data[lower_case(data[NAME]) + ":" + data[ID]]))
	    return;
	/* Set data[REQUEST] correctly, but still send to (req = REPLY) */
	data[REQUEST] = pending[REQUEST];
#ifdef INETD_DIAGNOSTICS
	data[PACKET_LOSS] = pending[PACKET_LOSS];
	data[RESPONSE_TIME] = time() - pending[RESPONSE_TIME] + 1;
#endif
#if 0
/* channel replies may not include a recipient, and shouldn't have one set */
	/* Restore the RECIPIENT in replies if none given and it is known. */
	if (!data[RECIPIENT] && pending[SENDER])
	    data[RECIPIENT] = pending[SENDER];
#endif
	pending_data =
	m_delete(pending_data, lower_case(data[NAME]) + ":" + data[ID]);
    }
    else if (data[ID]) {
	if (member_array(id = (lower_case(data[NAME]) + ":" + data[ID]),
	    received_ids) == -1)
	{
	    received_ids += ({ id });
	    call_out("remove_received_id",
	    REPLY_TIME_OUT + REPLY_TIME_OUT * (RETRIES + 1), id);
	}
	else
	    add_system_field(data, REPEAT);
    }
    if (!debug) err=catch(call_other(UDP_CMD_DIR + req, "udp_" + req,
        copy_mapping(data)));
    else {
        call_other(UDP_CMD_DIR + req, "udp_" + req, copy_mapping(data));
        err=0;
    }
    if (err)
    {
	send_udp(data[NAME], ([
	    REQUEST: REPLY,
	    RECIPIENT: data[SENDER],
	    ID: data[ID],
	    DATA: capitalize(req)+ " request failed @" + LOCAL_NAME + ".\n"
	]) );
	log_file(INETD_LOG_FILE, DATE + ": " + data[REQUEST] + " from " +
	  data[NAME] + " failed.\n  Error: " + err + "\n  Packet:\n  "+
          packet + "\n");
    }
}

int do_match(string mudname, string match_str) {
    return mudname[0..strlen(match_str)-1] == match_str;
}

string *expand_mud_name(string name) {
    return sort_array(
	filter_array(m_indices(hosts), #'do_match, name),
	#'>
    );
}

string encode(mixed arg) {
  if(objectp(arg))
    return file_name(arg);
  if(stringp(arg) && (arg[0] == '$' ||
    (string)to_int(arg) == (string)arg))
      return "$" + arg;
  return to_string(arg);
}

string encode_packet(mapping data) {
    int i;
    mixed indices;
    string header, body, t1, t2;
    string *ret;
    status data_flag;

    for(ret = ({ }), i = sizeof(indices = m_indices(data)); i--; ) {
	if (indices[i] == DATA) {
	    data_flag = 1;
	    continue;
	}
	header = encode(indices[i]);
	body = encode(data[indices[i]]);
	if (sscanf(header, "%s:%s", t1, t2) ||
	    sscanf(header + body, "%s" + DELIMITER + "%s", t1, t2)
	)
	    return 0;
	
	ret += ({ header + ":" + body });
    }
    if(data_flag)
      ret += ({ DATA + ":" + encode(data[DATA]) });
    return implode(ret, DELIMITER);
}

string *explode_packet(string packet, int len) {
    if (strlen(packet) <= len)
	return ({ packet });
    return ({ packet[0..len-1] }) +
    explode_packet( packet[len..], len);
}

varargs string send_udp(string mudname, mapping data, int expect_reply) {
    mixed host_data;
    string *packet_arr;
    string packet;
    int i;

    if (!mudname)
	return implode(map_array(m_indices(hosts),
	    #'send_udp, data, expect_reply)-({0}), "\n");
    mudname = lower_case(mudname);
    if (!(host_data = hosts[mudname])) {
	string *names;

	if (sizeof(names = expand_mud_name(mudname)) == 1)
	    host_data = hosts[mudname = names[0]];
	else
	    return "Unknown or ambiguous mudname: " + capitalize(mudname) + "\n";
    }
    if (data[REQUEST] != PING &&
    data[REQUEST] != QUERY &&
    data[REQUEST] != REPLY &&
    member_array("*", host_data[HOST_COMMANDS]) == -1 &&
    member_array(data[REQUEST], host_data[HOST_COMMANDS]) == -1)
	return capitalize(data[REQUEST]) + ": Command unavailable @" +
	host_data[HOST_NAME] + "\n";
    data[NAME] = LOCAL_NAME;
    data[UDP_PORT] = LOCAL_UDP_PORT;
    if (expect_reply) {
	/* Don't use zero. */
	data[ID] = ++packet_id;
	/* Don't need copy_mapping() as we are changing the mapping size. */
	pending_data[mudname + ":" + packet_id] =
#ifdef INETD_DIAGNOSTICS
	data + ([ NAME: host_data[HOST_NAME], RESPONSE_TIME: time() ]);
#else
	data + ([ NAME: host_data[HOST_NAME] ]);
#endif
    }
    if (!(packet = encode_packet(data))) {
	if (expect_reply)
	    pending_data = m_delete(pending_data, mudname + ":" + packet_id);
	log_file(INETD_LOG_FILE, DATE + ": Illegal packet sent by " +
	file_name(previous_object()) + "\n\n");
	return "inetd: Illegal packet.\n";
    }
    if (expect_reply)
	call_out("reply_time_out", REPLY_TIME_OUT, mudname + ":" + packet_id);
    if (strlen(packet) <= MAX_PACKET_LEN)
	packet_arr = ({ packet });
    else {
	string header;
	int max;

	/* Be careful with the ID.  data[ID] could have been set up by RETRY */
	header =
	    PACKET + ":" + lower_case(LOCAL_NAME) + ":" +
	    ((expect_reply || data[REQUEST] != REPLY)&& data[ID] ?
	    data[ID] : ++packet_id) + ":";
	/* Allow 8 extra chars: 3 digits + "/" + 3 digits + DELIMITER */
	packet_arr = explode_packet(packet,
	    MAX_PACKET_LEN - (strlen(header) + 8));
	for(i = max = sizeof(packet_arr); i--; )
	    packet_arr[i] =
	    header + (i+1) + "/" + max + DELIMITER + packet_arr[i];
    }
    for(i = sizeof(packet_arr); i--; ) {
	if (!send_imp(
	host_data[HOST_IP], host_data[HOST_UDP_PORT], packet_arr[i]))
	    return "inetd: Error in sending packet.\n";
    }
    return 0;
}

void reply_time_out(string id) {
    mapping data;

    if (data = pending_data[id]) {
	object ob;

#ifdef INETD_DIAGNOSTICS
	data[PACKET_LOSS]++;
#endif
	if (data[RETRY] < RETRIES) {
	    mapping send;

	    data[RETRY]++;
	    /* We must use a copy so the NAME field in pending_data[id]
	     * isn't corrupted by send_udp(). */
	    send = copy_mapping(data);
	    send = m_delete(send, RETRY);
#ifdef INETD_DIAGNOSTICS
	    send = m_delete(send, PACKET_LOSS);
	    send = m_delete(send, RESPONSE_TIME);
#endif
	    call_out("reply_time_out", REPLY_TIME_OUT, id);
	    send_udp(data[NAME], send);
	    return;
	}
	data = m_delete(data, RETRY);
#ifdef INETD_DIAGNOSTICS
	data = m_delete(data, RESPONSE_TIME);
#endif
	if (!debug) catch(call_other(UDP_CMD_DIR + REPLY, "udp_" + REPLY,
	    add_system_field(data, TIME_OUT)));
	else call_other(UDP_CMD_DIR + REPLY, "udp_" + REPLY,
	    add_system_field(data, TIME_OUT));
	/* It's just possible this was removed from the host list. */
        // why update DOWN-Time? (Holger)
        // if (hosts[lower_case(data[NAME])])
	//    hosts[lower_case(data[NAME])][HOST_STATUS] = DOWN;
	remove_incoming(lower_case(data[NAME]) + ":" + id);
    }
    pending_data = m_delete(pending_data, id);
}

void remove_received_id(string id) {
    received_ids -= ({ id });
}

varargs mixed query(string what, mixed extra1, mixed extra2) {
  mixed data;

  switch(what) {
    case "commands":
      return COMMANDS;
    case "hosts":
      return copy_mapping(hosts);
    case "pending":
      return copy_mapping(pending_data);
    case "incoming":
      return copy_mapping(incoming_packets);
    case "received":
      return ({ }) + received_ids;
    /* args: "valid_request", request, mudname */
    case "valid_request":
      if(data = hosts[extra2])
        return member_array("*", data[HOST_COMMANDS]) != -1 ||
          member_array(extra1, data[HOST_COMMANDS]) != -1;
      return 0;
  }
}

void reset() {
  if(counter<=0) {
    call_out("startup", 1);
    counter=PING_DELAY;
  }
  counter--;
}
  
int remove() {
  while(remove_call_out("remove_incoming") != -1);
  while(remove_call_out("remove_received_id") != -1);
  while(remove_call_out("reply_time_out") != -1);
  while(remove_call_out("startup") != -1);
  while(remove_call_out("check_mud") != -1);
  save_object(SAVEFILE);
  destruct(this_object());
  return 1;
}
#if 1
int ___toggle_debug() {
    // gesetztes 'debug' verhindert catch()
    // ist ein Objectp um versehentliches vergessen des Ausschaltens
    // zu verhindern
    if (debug) return debug=0;
    if (!this_interactive()) return -1;
    debug=this_interactive();
    return 1;
}
#endif
mapping usage() { return statistics; }
void print_usage() {
    mixed* names, erg, req, reqs, strs;
    int* sum, *a;
    int i, j, k;
    req=({"channel","finger","tell","who","mail","www","ping","man"});
    a=allocate(8+1);
    erg="Start: "+statistics["StartStatistics"]+"\nName               sum "
        "channel finger  tell  who  mail   www  ping   man  misc\n"
        "--------------------------------------------------------"
        "---------------------\n";
    names=m_indices(statistics)-({"StartStatistics"});
    j=sizeof(names);
    sum=allocate(j);
    for (; j--;) {
        reqs=statistics[names[j]]+([]);
        sum[j]=0;
        a[8]=0;
        for (k=7; k--;) {
	    a[k]=reqs[req[k]];
            sum[j]+=a[k];
            efun::m_delete(reqs, req[k]);
        }
        strs=m_indices(reqs);
        for (k=sizeof(strs); k--;) {
            a[8]+=reqs[strs[k]];
        }
        sum[j]+=a[8];
        names[j]=sprintf("%:-14s %:7d   %:5d %:5d %:5d %:5d %:5d %:5d %:5d "
            "%:5d %:5d\n", names[j], sum[j], a[0], a[1], a[2], a[3], a[4],
            a[5], a[6], a[7], a[8]);
    }
    names=sort_array(transpose_array(({sum, names})), lambda( ({ 'x, 'y }), ({
        #'>, ({ #'[, 'x, 0 }), ({ #'[, 'y, 0 }) }) ));
    for (j=sizeof(names); j--;) erg+=names[j][1];
    printf(erg);
}

string inetd_version() {
    string rcs;
    // Achtung, wird automatisch von RCS aktualisiert
    rcs="$Revision: 1.3 $";
    return "0.7wl"+rcs[strstr(rcs,": ")+2..<3];
}

