/*******************
** Eldarea MUDLib **
********************
**
** filename - short desc
**
** CVS DATA
** $Date: 1999/11/05 12:30:45 $
** $Revision: 1.1.1.1 $
**
** longdesc
**
** CVS History
**
** $Log: kal_master.c,v $
** Revision 1.1.1.1  1999/11/05 12:30:45  elatar
** Preparing mudlib for cvs control
**
**
*/
// M.D. Mudlib
// Basierend auf Wunderland Mudlib
//
// SECURE/KAL_MASTER.C --- Masterobjekt fuer die Kalenderverwaltung
//
// Author: Bongo@Wunderland
// Date  : 29.05.1997
//
// $Log: kal_master.c,v $
// Revision 1.1.1.1  1999/11/05 12:30:45  elatar
// Preparing mudlib for cvs control
//
// Revision 1.1.1.1  1999/11/04 12:48:12  en
// MUDLib CVS Preperation
//
// Revision 1.1  1999/08/05 12:13:58  Largo
// Initial revision
//

#include <properties.h>
#include <wizlevels.h>
#include <ansi.h>
#include <defines.h>
#include <service.h>

/******************************************/
/* Notwendige Konstanten werden definiert */
/******************************************/
// Mit diesem String ignoriert man alle Meldungen
#define IGNORE_ALL "geburtstage"

// Savefile fuer den Kalender
#define SAVEFILE "/etc/kalender"

// Playersoul fuer CountUp-Funktion
#define SOUL "/std/player/soul"

// Login-Master, fuer die Geburtstagsmeldungen
//#define LOGIN_OBJECTS LIB_SENSOR

// Maximale Textlaenge pro Eintrag
#define MAX_TEXT_LENGTH 100

// Maximale Anzahl der Eintraege pro Tag
#define MAX_ENTRIES 10

// Maximale Anzahl der Eintraege pro Person pro Tag
#define MAX_ENTRIES_PER_PERSON 3

// Ab hier werden die Daten fuer die Jahre ermittelt
#define BEGIN ({1996,0,1})  //  Jan 1996, 1.1. ist Montag, 1.1. ist 1. Woche
// Einige Schaltjahre (muessen bei Bedarf per Hand gesetzt werden)
#define SCHALTJAHRE ({1996,2000,2004})

// Anzahl der Tage im Monat
#define DAYS_PER_MONTH ({31,28,31,30,31,30,31,31,30,31,30,31})

// Kuerzel fuer Wochentage (identisch zu ctime())
#define DAYS ({"Mon","Tue","Wed","Thu","Fri","Sat","Sun"})

// Monatsnamen
#define MON_NAMES ({    \
                  "                                J A N U A R",      \
                  "                               F E B R U A R",     \
                  "                                 M A E R Z",        \
                  "                                 A P R I L",        \
                  "                                   M A I",          \
                  "                                  J U N I",         \
                  "                                  J U L I",         \
                  "                                A U G U S T",       \
                  "                             S E P T E M B E R",    \
                  "                               O K T O B E R",      \
                  "                              N O V E M B E R",     \
                  "                              D E Z E M B E R" })

// Blanks fuer mittiges Layout
#define BLANKS  "                                  "

// Farben im Kalender (koennen auch selbst definiert werden)
#define KAL1 "kalender-ueberschriften"
#define KAL2 "kalender-heute"
#define KAL3 "kalender-tage"
#define KAL4 "geburtstagsmeldung"
#define KAL5 "kalender-sonntage"

// Laenge fuer break_string
#define WD BS_STDLEN

/*******************************/
/* Objektvariablen deklarieren */
/*******************************/
int *years;
mapping months_in_year;
mapping entries;
mapping birthdays;


/*************************************/
/* Verwendete Funktionen deklarieren */
/*************************************/
void init_kalender();
int *get_date();
int *valid_date(string date, int with_year);
string get_month_name(int month);
string get_date_str(mixed date);
int check_month(mixed month);
int check_year(int year);
int registration(int day, int month, int year);
int AddEntry(int year, int month, int day, string name, string text);
int RemoveEntry(int year, int month, int day, string name, int number);
int AddBirthday(int month, int day, string name);
int RemoveBirthday(string name);
string GetMonth(string month, int year);
mixed GetDay(int year, int month, int day);
int check_for_birthday(object player);
string ListAllBirthdays();
string ListAllEntries();


void create()
{
  seteuid(getuid());

  if (clonep())
    destruct(this_object());

  if (file_size(SAVEFILE+".o") > -1) 
    restore_object(SAVEFILE);
  else
  {
    entries = ([]);
    birthdays = ([]);
  }

  init_kalender();
  //LOGIN_OBJECTS->AddLoginObject(this_object(), "check_for_birthday");
}


int *get_years()
{
  return years;
}


mapping get_months_in_year()
{
  return months_in_year;
}


mapping get_entries()
{
  return entries;
}


mapping get_birthdays()
{
  return birthdays;
}


/************************************************************************/
/*                                                                      */
/* void init_kalender()                                                 */
/*                                                                      */
/* Diese Funktion erstellt immer 3 Jahre fuer den Kalender. Dafuer wird */
/* folgendes Mapping angelegt:                                          */
/*                                                                      */
/* ([ jahr1 : ([ monat1 : ({ starttag , tage , woche }),                */
/*               monat2 : ....... ]),                                   */
/*    jahr2 : ([ monat1 : ({ starttag , tage , woche }),                */
/*               monat2 : ....... ]),                                   */
/*    jahr3 : ([ monat1 : ({ starttag , tage , woche }),                */
/*               monat2 : ....... ]) ])                                 */
/*                                                                      */
/* jahr1   : Das gerade vergangene Jahr (bsp: 1996)                     */
/* jahr2   : Das aktuelle Jahr (bsp: 1997)                              */
/* jahr3   : Das naechste Jahr (bsp: 1998)                              */
/* monat1  : Erster Monat im Jahr (Januar) ist identisch mit Index 0    */
/*            (Januar = 0 ... Dezember = 11)                            */
/* starttag: Wochentag des ersten Tages dieses Monats                   */
/*            (Montag = 0 ... Sonntag = 6)                              */
/* tage    : Anzahl der Tage in diesem Monat (Schaltjahre werden        */
/*           beruecksichtigt)                                           */
/* woche   : Kalenderwoche am ersten des Monats                         */
/*                                                                      */
/* Anhand dieser Daten kann nun jederzeit mitten in den Kalender        */
/* hineingegriffen werden, um einen bestimmten Monat oder einen Tag     */
/* zu extrahieren.                                                      */
/*                                                                      */
/************************************************************************/
void init_kalender() {
  int i, now_year, start_day, stop_day, year;
  int mon_tage, w_zahl, start_week, stop_week;

  months_in_year = ([]);

  // Aktuelles Jahr wird ermittelt
  year = get_date()[0];

  stop_day = BEGIN[1]-1;
  stop_week = BEGIN[2]-1;

  for (now_year=BEGIN[0]; now_year<=year+1; now_year++) {
    if (abs(now_year-year)<2) // Vorjahr, aktuelles, naechstes Jahr
      months_in_year += ([ now_year: ([]) ]);
     
    // Es wird jeder Monat im Jahr durchgearbeitet
    for (i=0; i<12; i++) {

      // Anzahl der Tage im Monat wird ermittelt
      // (Schaltjahre werden beruecksichtigt)
      mon_tage = DAYS_PER_MONTH[i];
      if (i==1 && member(SCHALTJAHRE, now_year)!=-1) mon_tage++;

      // Wochentag von Monatsbeginn und -ende
      start_day = (stop_day+1)%7;
      stop_day = (start_day+(mon_tage%7)+6)%7;
      
      // In welcher Woche liegt der erste und letzte des Monats?
      if (i==0 && start_day>3) { // erste Woche gehoert noch zu letztem Jahr
        start_week=stop_week;
        stop_week=(mon_tage+start_day-1)/7;
      } else {
        start_week=stop_week;
        if (start_day==0) start_week++; // montags startet ne neue Woche
        if (i==0) start_week=1;
        stop_week=start_week+(mon_tage+start_day-1)/7;
      }
        
      // Werte werden gegebenenfalls in das Mapping eingetragen
      if (abs(now_year-year)<2)
        months_in_year[now_year] +=
          ([ i: ({ start_day , mon_tage , start_week }) ]);
    }
  }

  save_object(SAVEFILE);
}


/************************************************************************/
/*                                                                      */
/* int check_month(mixed month)                                         */
/*                                                                      */
/* Diese Funktion prueft, ob der uebergebene Wert fuer einen Monat      */
/* Gueltigkeit besitzt. Gueltige Werte sind die gaengigen deutschen und */
/* amerikanischen Kuerzel, die ausgeschriebenen Monate und die Nummer   */
/* des Monats. Alle Werte sind case-insensitiv.                         */
/*                                                                      */
/* Beispiel:                                                            */
/*   month == "dez"                                                     */
/*   month == "dezember"                                                */
/*   month == "dec"                                                     */
/*   month == "december"                                                */
/*   month == 12                                                        */
/*                                                                      */
/*   Bei Januar ist beispielsweise auch der Wert 01 gueltig.            */
/*                                                                      */
/* Zurueckgegeben wird dann der Index des Monats, der der um einen      */
/* verringerten Monatsnummer entspricht.                                */
/*                                                                      */
/* Returnwerte:                                                         */
/*    -1 : Falsche Angabe fuer einen Monat                              */
/*   ret : Index des Monats (Januar = 0 .. Dezember == 11)              */
/*                                                                      */
/************************************************************************/
int check_month(mixed month)
{
  int m;
  string mstr;

  string *MONTHS1, *MONTHS2, *MONTHS3, *MONTHS4;

  MONTHS1 = ({"jan","feb","mar","apr","may","jun",
              "jul","aug","sep","oct","nov","dec"});
  MONTHS2 = ({"january","february","march","april","may","june",
              "july","august","september","october","november","december"});
  MONTHS3 = ({"jan","feb","mae","apr","mai","jun",
              "jul","aug","sep","okt","nov","dez"});
  MONTHS4 = ({"januar","februar","maerz","april","mai","juni",
              "juli","august","september","oktober","november","dezember"});

  mstr = to_string(month);

  if (mstr)
  {
    mstr = lower_case(mstr);

    if ((m=member(MONTHS1,mstr))==-1 && (m=member(MONTHS2,mstr))==-1 &&
        (m=member(MONTHS3,mstr))==-1 && (m=member(MONTHS4,mstr))==-1)
      m = -1;
  }

  if (m<0)
  {
    m = to_int(month) - 1;

    if (m==-2)
      m = get_date()[1];
    if (m<=-1 || m>11)
      m = -1;
  }

  return m;
}


/************************************************************************/
/*                                                                      */
/* int check_year(int year)                                             */
/*                                                                      */
/* Diese Funktion prueft, ob das vorgegebene Jahr ein gueltiges und     */
/* vom Kalender verwaltetes Jahr ist. Es muss also eines der drei       */
/* ermittelten Jahre sein. Es sind folgende Kuerzungen moeglich:        */
/*                                                                      */
/* year == 0 --> 1997  es wird das aktuelle Jahr zurueckgegeben         */
/* year == 1 --> 1901  wird nicht verwaltet, also ungueltig             */
/* year == 97 -> 1997  wird verwaltet, also gueltig                     */
/* year == 1997        wird verwaltet, also gueltig                     */
/*                                                                      */
/* Returnwerte:                                                         */
/*    -1 : Falsche Angabe fuer ein Jahr                                 */
/*   ret : Das ermittelte Jahr wird zurueckgegeben                      */
/*                                                                      */
/************************************************************************/
int check_year(int year)
{
  if (!intp(year))
    return -1;

  if (year<=0)
    year = get_date()[0];

  if (year>-1 && year<100)
    year += 1900;

  if (member(years,year)==-1)
    return -1;

  return year;
}


/******************************************************************************/
/*                                                                            */
/* string GetMonth(string month, int year)                                    */
/*                                                                            */
/* Diese Funktion wird von aussen aufgerufen und stellt den angegebenen       */
/* Monat tabellarisch dar. Existieren Eintragungen oder Geburtstage           */
/* zu einem der Tage, so wird dieser Tag gekennzeichnet. Das aktuelle         */
/* Datum und Sonntage werden auf Ansi- und VT100-Terminals besonders          */
/* dargestellt.                                                               */
/* Der so komplett zusammengesetzte String wird am Ende an das                */
/* aufrufende Objekt zurueckgegeben.                                          */
/*                                                                            */
/* Beispiel:                                                                  */
/*                                    M A I                                   */
/*                                                                            */
/*                                   1 9 9 7                                  */
/*                                                                            */
/* +------------------------------------------------------------------------+ */
/* | Woche  ||   Mo   |   Di   |   Mi   |   Do   |   Fr   |   Sa   |   So   | */
/* |--------++--------+--------+--------+--------+--------+--------+--------| */
/* |        ||        |        |        |        |        |        |        | */
/* |   18   ||        |        |        |   1    |   2    |   3    |   4    | */
/* |        ||        |        |        |        |        |        |        | */
/* |--------++--------+--------+--------+--------+--------+--------+--------| */
/* |        ||        |        |        |        |        |        |        | */
/* |   19   ||   5    |   6    |   7    |   8    |   9    |   10   |   11   | */
/* |        ||        |        |        |        |        |        |        | */
/* |--------++--------+--------+--------+--------+--------+--------+--------| */
/* |        ||        |        |        |        |        |        |        | */
/* |   20   ||   12   |   13   |_  14   |_  15   |   16   |   17   |   18   | */
/* |        ||        |        |*|      |*|      |        |        |        | */
/* |--------++--------+--------+--------+--------+--------+--------+--------| */
/* |        ||        |        |        |        |        |        |        | */
/* |   21   ||   19   |   20   |   21   |   22   |   23   |   24   |   25   | */
/* |        ||        |        |        |        |        |        |        | */
/* |--------++--------+--------+--------+--------+--------+--------+--------| */
/* |        ||        |        |        |        |        |        |        | */
/* |   22   ||   26   |   27   |   28   |   29   |   30   |_  31   |        | */
/* |        ||        |        |        |        |        |*|      |        | */
/* +------------------------------------------------------------------------+ */
/*                                                                            */
/* Parameter:                                                                 */
/*   month: Der gesuchte Monat (kann alle Versionen aus check_month()         */
/*          annehmen)                                                         */
/*   year : Das gesuchte Jahr (kann alle Versionen aus check_year()           */
/*          annehmen)                                                         */
/*                                                                            */
/******************************************************************************/
string GetMonth(string month, int year)
{
  int *h, i, k, m, pointer, counter, first, max_days, help, week;
  string y, text, text2;
  object player;

  year = check_year(year);
  m = check_month(month);

  if (year<0)
    return "Dieses Jahr ist nicht im Kalender enthalten.\n";

  if (m<0)
    return "So einen Monat gibt es nicht.\n";

  y = to_string(year);
  y = y[0..0]+" "+y[1..1]+" "+y[2..2]+" "+y[3..3];

  // Anfangswerte fuer den Monat werden aus dem Mapping eingelesen
  first = months_in_year[year][m][0];
  max_days = months_in_year[year][m][1];
  week = months_in_year[year][m][2];

  player = this_player();

  h = get_date();
  pointer = 0;
  counter = 1;
  k = 0;

  text = "\n"+player->in_color(KAL1, MON_NAMES[m]+"\n\n"+BLANKS+y)+"\n\n";
  text += "+------------------------------------------------------------------------+\n"
         +"| Woche  ||   Mo   |   Di   |   Mi   |   Do   |   Fr   |   Sa   |   So   |\n"
         +"|--------++--------+--------+--------+--------+--------+--------+--------|\n";

  // Wochen werden zusammengestellt, bis der Monat zuende ist
  while (!pointer)
  {
    text += "|        ||        |        |        |        |        |        |        |\n|";
    text2 = "|        ||";

    text += sprintf("   %2d   ||", week);

    // Tage werden zusammengestellt
    for (i=0;i<7;i++)
    {
      if (k<first)
      {
        text += "        |";
        text2 += "        |";
      }
      else if (k>=first && counter<=max_days)
      {
        help = registration(counter,m,year);

        if (h[0]==year && h[1]==m && h[2]==counter)
          text += sprintf("%s  %s   |", (help?"_":" "),
            player->in_color(KAL2, sprintf("%2d",counter)));
        else
        {
          text += sprintf("%s  %s   |", (help?"_":" "),
            player->in_color((i>5?KAL5:KAL3), sprintf("%2d",counter)));

/*  Wieso ist das hier auskommentiert?  Fi
          if (i>5)
            text += sprintf("%s  %s   |", (help?"_":" "),
              player->in_color(KAL5, sprintf("%2d",counter)));
          else
            text += sprintf("%s  %2d   |", (help?"_":" "), counter);
*/
        }

        if (help)
          text2 += "*|      |";
        else
          text2 += "        |";

        if (counter==max_days)
          pointer = 1;

        counter++;
      }
      else if (counter>=max_days)
      {
        text += "        |";
        text2 += "        |";
        pointer = 1;
        counter++;
      }

      k++;
    }

    text += "\n"+text2+"\n";
    week++;
    if (m==0 && week>10) week=1; // fuer ersten Januartage in alter Woche

    if (pointer)
      text += "+------------------------------------------------------------------------+\n";
    else
      text += "|--------++--------+--------+--------+--------+--------+--------+--------|\n";
  }

  return text;
}


/************************************************************************/
/*                                                                      */
/* mixed GetDay(int year, int month, int day)                           */
/*                                                                      */
/* Diese Funktion wird von aussen aufgerufen und ermittelt alle         */
/* Eintraege und Geburtstage an einem angegebenen Tag.                  */
/* Die Namen der Geburtstagskinder und der Eintraeger werden sortiert   */
/* ausgegeben. Die Eintraege werden je Name durchnummeriert. Mit diesen */
/* Nummern koennen die Eintraege auch wieder geloescht werden.          */
/*                                                                      */
/* Parameter:                                                           */
/*   year : Das gesuchte Jahr (kann alle Versionen aus check_year()     */
/*          annehmen)                                                   */
/*   month: Der gesuchte Monat (kann alle Versionen aus check_month()   */
/*          annehmen)                                                   */
/*   day  : Der gesuchte Tag                                            */
/*                                                                      */
/* Returnwerte:                                                         */
/*      0 : Ungueltiges Jahr oder ungueltiger Monat wurden angegeben    */
/*     -1 : Ungueltiger Tag wurde angegeben                             */
/*     -2 : Keine Eintraege fuer den angegebenen Tag                    */
/*   text : Zusammengesetzter String mit allen Informationen            */
/*                                                                      */
/************************************************************************/
mixed GetDay(int year, int month, int day)
{
  int p1, p2, i, j, k, l, y, m, d;
  string *values, *values2, text;
  object player;

  // Datum ueberpruefen
  y = check_year(year);
  m = check_month(month);

  if (y<0 || m<0)
    return 0;

  d = months_in_year[y][m][1];

  if (day<0 || day>d)
    return -1;

  d = day;
  p2 = 0;

  player = this_player();

  text = "\n---------------------------------------------------"
        +"---------------------------\n";
  text += sprintf("%s %d. %s %d\n%s\n",
          player->in_color(KAL1, "Datum:"), d, get_month_name(m), y,
          player->in_color(KAL1, "------"));

  // Geburtstage suchen
  p1 = member(birthdays, m);
  p1 = (p1?member(birthdays[m],d):0);

  if (p1)
  {
    values = sort_array(birthdays[m][d], #'>);

    for (j=sizeof(values),i=0;i<j;i++)
      values[i] = capitalize(values[i]);

    text += sprintf("\n%s\n%s",
            player->in_color(KAL1, "Geburtstage:\n------------"),
            break_string(SOUL->CountUp(values),WD,2));
    p2 = 1;
  }

  // Eintraege suchen
  p1 = member(entries, y);
  p1 = (p1?member(entries[y],m):0);
  p1 = (p1?member(entries[y][m],d):0);

  if (p1)
  {
    // Eintraege zusammenstellen
    values = sort_array(m_indices(entries[y][m][d]), #'>);

    text += sprintf("\n%s\n",
            player->in_color(KAL1, "Eintragungen:\n-------------"));

    for (j=sizeof(values),i=0;i<j;i++)
    {
      values2 = entries[y][m][d][values[i]];

      text += sprintf("  %s:\n", capitalize(values[i]));

      for (l=sizeof(values2),k=0;k<l;k++)
//        text += break_string(sprintf("--> %d.) %s", k+1,
        text += break_string(sprintf("%s", entries[y][m][d][values[i]][k]),
                WD, "    "+to_string(k+1)+".) ", 1);
    }
  }
  else if (!p2)
    return -2;

  text += "-------------------------------------------"
         +"-----------------------------------\n";

  return text;
}


/************************************************************************/
/*                                                                      */
/* int registration(int day, int month, int year)                       */
/*                                                                      */
/* Diese Funktion prueft, ob der angegebene Tag irgendwelche            */
/* Eintragungen oder Geburtstage hat.                                   */
/*                                                                      */
/* Parameter:                                                           */
/*   day  : Der gesuchte Tag                                            */
/*   month: Der gesuchte Monat (kann alle Versionen aus check_month()   */
/*          annehmen)                                                   */
/*   year : Das gesuchte Jahr (kann alle Versionen aus check_year()     */
/*          annehmen)                                                   */
/*                                                                      */
/* Returnwerte:                                                         */
/*      0 : Der Tag enthaelt keinerlei Eintragungen oder Geburtstage    */
/*      1 : Der Tag enhaelt irgendwelche Eintragungen oder Geburtstage  */
/*                                                                      */
/************************************************************************/
int registration(int day, int month, int year)
{
  int p1, p2, y, m, d;

  // Datum ueberpruefen
  y = check_year(year);
  m = check_month(month+1);

  if (y<0 || m<0)
    return 0;

  d = months_in_year[y][m][1];

  if (day<0 || day>d)
    return 0;

  d = day;

  // Gibt es Eintragungen?
  p1 = member(entries, y);
  p1 = (p1?member(entries[y],m):0);
  p1 = (p1?member(entries[y][m],d):0);
  p2 = 0;

  if (p1 && m_sizeof(entries[y][m][d]))
    p2 = 1;

  // Gibt es Geburtstage?
  p1 = member(birthdays, m);
  p1 = (p1?member(birthdays[m],d):0);

  if (p1 && sizeof(birthdays[m][d]))
    p2 = 1;

  if (p2)
    return 1;

  return 0;
}


/*****************************************************************************/
/*                                                                           */
/* int AddEntry(int year, int month, int day, string name, string text)      */
/*                                                                           */
/* Mit dieser Funktion kann eine Eintragung fuer einen bestimmten Tag        */
/* gemacht werden. Dabei wird folgendes Mapping aufgebaut:                   */
/*                                                                           */
/* ([jahr1:([monat1:([tag1:([name1:({text1,text2,...}),...]),...]),...]),...]*/
/*                                                                           */
/* jahr1 : Das Jahr, fuer das der Eintrag gemacht wird                       */
/* monat1: Der Monat, fuer den der Eintrag gemacht wird                      */
/* tag1  : Der Tag , fuer den der Eintrag gemacht wird                       */
/* name1 : Der Name des Eintraegers                                          */
/* text1 : Der Eintrag des Spielers                                          */
/*                                                                           */
/* Parameter:                                                                */
/*   year : Das Jahr in das der Eintrag soll                                 */
/*   month: Der Monat, in den der Eintrag soll                               */
/*   day  : Der Tag, in den der Eintrag soll                                 */
/*   name : Der Name, unter dem der Eintrag vermerkt wird (in der Regel der  */
/*          Spielername)                                                     */
/*   text : Der Eintrag, der gemerkt werden soll                             */
/*                                                                           */
/* Returnwerte:                                                              */
/*    0 : Falsche Parameter                                                  */
/*   -1 : Falsche Werte fuer das Jahr oder den Monat                         */
/*   -2 : Falscher Wert fuer den Tag                                         */
/*   -3 : Datum liegt vor dem heutigen Tage (Eintragung sinnlos)             */
/*   -4 : Nur Erzmagier duerfen fremde Namen eintragen                       */
/*   -5 : Der Text darf eine Laenge von MAX_TEXT_LENGTH nicht ueberschreiten */
/*   -6 : Es duerfen nur MAX_ENTRIES pro Tag eingetragen werden              */
/*   -7 : Jeder Name darf nur MAX_ENTRIES_PER_PERSON je Tag eintragen        */
/*    1 : Eintragung durchgefuehrt und Mapping gespeichert                   */
/*                                                                           */
/*****************************************************************************/
int AddEntry(int year, int month, int day, string name, string text)
{
  int *ret, p1, p2, p3, p4, y, m, d, now_day, now_mon, now_year;
  object player;

  if (!name || !stringp(name) || !text || !stringp(text))
    return 0;

  // Wurde ein gueltiges Datum angegeben?
  y = check_year(year);
  m = check_month(month);

  if (y<0 || m<0)
    return -1;

  d = months_in_year[y][m][1];

  if (day<0 || day>d)
    return -2;

  d = day;
  ret = get_date();
  now_year = ret[0];
  now_mon = ret[1];
  now_day = ret[2];

  if (now_year>y)
    return -3;
  else if (now_year==y && now_mon>m)
    return -3;
  else if (now_year==y && now_mon==m && now_day>d)
    return -3;

  player = this_player();

  // Nur Erzmagier duerfen unter einem falschen Namen eintragen
  if (!IS_ARCH(player) && player->query_real_name()!=name)
    return -4;

  // Die Texte duerfe nicht zu gross werden
  if (strlen(text)>MAX_TEXT_LENGTH)
    return -5;

  p1 = member(entries, y);
  p2 = (p1?member(entries[y],m):0);
  p3 = (p2?member(entries[y][m],d):0);
  p4 = (p3?member(entries[y][m][d],name):0);

  // Es duerfen nur begrenzte Eintraege gemacht werden
  if (p3 && m_sizeof(entries[y][m][d])>MAX_ENTRIES)
    return -6;

  // Es duerfen nur begrenzte Eintraege pro Person gemacht werden
  if (p4 && sizeof(entries[y][m][d][name])>MAX_ENTRIES_PER_PERSON)
    return -7;

  // Eintragung vornehmen
  if (p4)
    entries[y][m][d][name] += ({ text });
  else if (p3)
    entries[y][m][d] += ([ name : ({ text }) ]);
  else if (p2)
    entries[y][m] += ([ d : ([ name : ({ text }) ]) ]);
  else if (p1)
    entries[y] += ([ m : ([ d : ([ name : ({ text }) ]) ]) ]);
  else
    entries += ([ y : ([ m : ([ d : ([ name : ({ text }) ]) ]) ]) ]);

  save_object(SAVEFILE);
  return 1;
}


/*****************************************************************************/
/*                                                                           */
/* int RemoveEntry(int year, int month, int day, string name, int number)    */
/*                                                                           */
/* Mit dieser Funktion kann eine Eintragung fuer einen bestimmten Tag        */
/* wieder geloescht werden. Dazu ist nur der Spieler mit dem entsprechenden  */
/* Namen oder aber ein Erzmagier befaehigt.                                  */
/*                                                                           */
/* Parameter:                                                                */
/*   year : Das Jahr in dem der Eintrag steht                                */
/*   month: Der Monat, in dem der Eintrag steht                              */
/*   day  : Der Tag, in dem der Eintrag steht                                */
/*   name : Der Name, unter dem der Eintrag vermerkt ist (in der Regel der   */
/*          Spielername)                                                     */
/*   text : Der Index des Eintrages, der geloescht werden soll (beginnt bei 0*/
/*                                                                           */
/* Returnwerte:                                                              */
/*    0 : Falsche Parameter                                                  */
/*   -1 : Falsche Werte fuer das Jahr oder den Monat                         */
/*   -2 : Falscher Wert fuer den Tag                                         */
/*   -3 : Nur Erzmagier duerfen fremde Namen loeschen                        */
/*   -4 : Text unter der gegeben Nummer nicht gefunden                       */
/*    1 : Austragung durchgefuehrt und Mapping gespeichert                   */
/*                                                                           */
/*****************************************************************************/
int RemoveEntry(int year, int month, int day, string name, int number)
{
  int p1, p2, p3, p4, y, m, d;
  object player;

  if (!name || !stringp(name) || !intp(number) || number<0)
    return 0;

  // Wurde ein gueltiges Datum angegeben?
  y = check_year(year);
  m = check_month(month);

  if (y<0 || m<0)
    return -1;

  d = months_in_year[y][m][1];

  if (day<0 || day>d)
    return -2;

  d = day;
  player = this_player();

  // Nur Erzmagier duerfen jeden Eintrag loeschen
  if (!IS_ARCH(player) && player->query_real_name()!=name)
    return -3;

  p1 = member(entries, y);
  p2 = (p1?member(entries[y],m):0);
  p3 = (p2?member(entries[y][m],d):0);
  p4 = (p3?member(entries[y][m][d],name):0);

  
  // Hat der name ueberhaupt einen Text mit der Nummer?
  if (!p4 || (sizeof(entries[y][m][d][name])-1)<number)
    return -4;

  // Mappings zurueckstutzen und gegebenenfalls loeschen
  entries[y][m][d][name] -= ({ entries[y][m][d][name][number] });

  if (!sizeof(entries[y][m][d][name]))
  {
    entries[y][m][d] = m_delete(entries[y][m][d], name);

    if (!m_sizeof(entries[y][m][d]))
    {
      entries[y][m] = m_delete(entries[y][m], d);

      if (!m_sizeof(entries[y][m]))
      {
        entries[y] = m_delete(entries[y], m);

        if (!m_sizeof(entries[y]))
          entries = m_delete(entries, y);
      }
    }
  }

  save_object(SAVEFILE);
  return 1;
}


/*****************************************************************************/
/*                                                                           */
/* int AddBirthday(int month, int day, string name)                          */
/*                                                                           */
/* Mit dieser Funktion kann der Geburtstag eines Spielers an einem bestimmten*/
/* Tag eingetragen werden. Dabei wird folgendes Mapping aufgebaut:           */
/*                                                                           */
/* ([ monat1 : ([ tag1 : ({ name1, name2, name3, ... }) ]) ])                */
/*                                                                           */
/* monat1: Der Monat, fuer den der Eintrag gemacht wird                      */
/* tag1  : Der Tag , fuer den der Eintrag gemacht wird                       */
/* name1 : Die Namen der Spieler, die an diesem Tage Geburtstage haben       */
/*                                                                           */
/* Parameter:                                                                */
/*   month: Der Monat, in den der Eintrag soll                               */
/*   day  : Der Tag, in den der Eintrag soll                                 */
/*   name : Der Name des Spielers, der an diesem Tage Geburtstag hat         */
/*                                                                           */
/* Returnwerte:                                                              */
/*    0 : Falsche Parameter                                                  */
/*   -1 : Falscher Wert fuer den Monat                                       */
/*   -2 : Falscher Wert fuer den Tag                                         */
/*   -3 : Nur Erzmagier duerfen fremde Namen eintragen                       */
/*   -4 : Der Name ist schon mit einem Geburtstag eingetragen                */
/*    1 : Eintragung durchgefuehrt und Mapping gespeichert                   */
/*                                                                           */
/*****************************************************************************/
int AddBirthday(int month, int day, string name)
{
  int p1, p2, p3, m, d, mon_s, day_s, i, j;
  string *a_mon, *a_day;
  object player;

  if (!name || !stringp(name))
    return 0;

  // Wurde ein gueltiges Datum angegeben?
  m = check_month(month);

  if (m<0)
    return -1;

  d = (m==1?DAYS_PER_MONTH[m]+1:DAYS_PER_MONTH[m]);

  if (day<0 || day>d)
    return -2;

  d = day;

  player = this_player();

  if (!IS_ARCH(player) && player->query_real_name()!=name)
    return -3;

  // Komplettes Mapping durchsuchen
  a_mon = m_indices(birthdays);

  for (mon_s=sizeof(a_mon),i=0;i<mon_s;i++)
  {
    a_day = m_indices(birthdays[a_mon[i]]);

    for (day_s=sizeof(a_day),j=0;j<day_s;j++)
    {
      // Name ist schon im Mapping enthalten
      if (member(birthdays[a_mon[i]][a_day[j]],name)!=-1)
        return -4;

      // Leere Keys werden geloescht 
      if (!sizeof(birthdays[a_mon[i]][a_day[j]]))
        birthdays[a_mon[i]] = m_delete(birthdays[a_mon[i]], a_day[j]);
    }

    // Wenn man schon alles durchsucht, kann man auch gleich mit aufraeumen
    if (!m_sizeof(birthdays[a_mon[i]]))
      birthdays = m_delete(birthdays, a_mon[i]);
  }

  // Eintragung wird vorgenommen
  p1 = member(birthdays, m);
  p2 = (p1?member(birthdays[m],d):0);

  if (p2)
    birthdays[m][d] += ({ name });
  else if (p1)
    birthdays[m] += ([ d : ({ name }) ]);
  else
    birthdays += ([ m : ([ d : ({ name }) ]) ]);

  save_object(SAVEFILE);
  return 1;
}


/*****************************************************************************/
/*                                                                           */
/* int RemoveBirthday(string name)                                           */
/*                                                                           */
/* Mit dieser Funktion kann der Geburtstag eines Spielers an einem bestimmten*/
/* Tag wieder ausgetragen werden. Dazu ist entweder der entsprechende Spieler*/
/* oder ein Erzmagier befaehigt.                                             */
/*                                                                           */
/* Parameter:                                                                */
/*   name : Der Name des Spielers, der ausgetragen werden soll               */
/*                                                                           */
/* Returnwerte:                                                              */
/*    0 : Falsche Parameter                                                  */
/*   -1 : Nur Erzmagier duerfen fremde Namen loeschen                        */
/*   -2 : Keine Geburtstage im Mapping enthalten                             */
/*   -3 : Der Geburtstag des Spielers ist nicht eingetragen                  */
/*    1 : Austragung durchgefuehrt und Mapping gespeichert                   */
/*                                                                           */
/*****************************************************************************/
int RemoveBirthday(string name)
{
  int i, j, mon_s, day_s, p;
  string *day, *mon;
  object player;

  if (!name || !stringp(name))
    return 0;

  player = this_player();

  if (!IS_ARCH(player) && player->query_real_name()!=name)
    return -1;

  // Mapping durchsuchen und name austragen
  mon = m_indices(birthdays);

  if (!(mon_s=sizeof(mon)))
    return -2;

  p = 0;

  for (i=0;i<mon_s;i++)
  {
    day = m_indices(birthdays[mon[i]]);

    for (day_s=sizeof(day),j=0;j<day_s;j++)
    {
      if (member(birthdays[mon[i]][day[j]],name)!=-1)
      {
        birthdays[mon[i]][day[j]] -= ({ name });
        p = 1;
      }

      if (!sizeof(birthdays[mon[i]][day[j]]))
        birthdays[mon[i]] = m_delete(birthdays[mon[i]], day[j]);
    }

    if (!m_sizeof(birthdays[mon[i]]))
      birthdays = m_delete(birthdays, mon[i]);
  }

  save_object(SAVEFILE);

  if (!p)
    return -3;

  return 1;
}


/************************************************************************/
/*                                                                      */
/* int *get_date()                                                      */
/*                                                                      */
/* Diese Funktion ermittelt das aktuelle Datum und gibt folgendes Array */
/* zurueck:                                                             */
/*                                                                      */
/* ({ aktuelles_jahr , aktueller_monat , aktueller_tag })               */
/*                                                                      */
/************************************************************************/
int *get_date()
{
  int now_day, now_mon, now_year;
  string time_str;

  time_str = ctime(time());
  now_day = to_int(time_str[8..9]);
  now_mon = check_month(time_str[4..6]);
  now_year = to_int(time_str[<4..]);

  return ({now_year, now_mon, now_day});
}


/************************************************************************/
/*                                                                      */
/* int *valid_date(string date, with_year)                              */
/*                                                                      */
/* Diese Funktion prueft, ob ein uebergebener String zu einem korrekten */
/* Datum umgewandelt werden kann. Folgender Array wird zurueckgegeben:  */
/*                                                                      */
/* ({ jahr , monat , tag })           oder (bei with_year==0)           */
/* ({ monat , tag })                                                    */
/*                                                                      */
/* with_year : Kennung, ob das Jahr wichtig ist.                        */
/*             with_year = 0 : 29.2.yyyy ist immer gueltig              */
/*             with_year = 1 : 29.2.yyyy ist nur guelig, wenn yyyy ein  */
/*                             Schaltjahr ist                           */
/*                                                                      */
/* Returnwerte:                                                         */
/*   ({ -1 }) : Kein Datum wurde angegeben                              */
/*   ({ -2 }) : Datum hat das falsche Format                            */
/*   ({ -3 }) : Falsche Angabe fuer das Jahr. Jahr existiert nicht oder */
/*              wird nicht vom Kalender verwaltet                       */
/*   ({ -4 }) : Der Monat ist nicht richtig angegeben worden            */
/*   ({ -5 }) : Der Tag ist nicht gueltig fuer diesen Monat             */
/*                                                                      */
/************************************************************************/
int *valid_date(string date, int with_year)
{
  int year, d, m, y;
  string *parts;
  mixed month;

  if (!date || !stringp(date))
    return ({ -1 });

  if (with_year)
  {
    parts = regexplode(date, "[/.,:; ]");
    parts = parts - ({"/"}) - ({""}) - ({"."}) - ({","}) - ({":"}) - ({";"}) - ({" "});

    switch(sizeof(parts))
    {
      case 0:
        return ({ -2 });
      case 1:
        d = to_int(parts[0]);
        month = -1;
        year = -1;
        break;
      case 2:
        d = to_int(parts[0]);
        month = parts[1];
        year = -1;
        break;
      case 3:
        d = to_int(parts[0]);
        month = parts[1];
        year = to_int(parts[2]);
        break;
      default:
        return ({ -2 });
    }

    y = check_year(year);
    m = check_month(month);

    if (y<0)
      return ({ -3 });

    if (m<0)
      return ({ -4 });

    if (d<=0 || d>months_in_year[y][m][1])
      return ({ -5 });

    return ({y, m, d});
  }
  else
  {
    parts = regexplode(date, "[/.,:; ]");
    parts = parts - ({"/"}) - ({""}) - ({"."}) - ({","}) - ({":"}) - ({";"}) - ({" "});

    switch(sizeof(parts))
    {
      case 0:
        return ({ -2 });
      case 1:
        d = to_int(parts[0]);
        month = -1;
        break;
      case 2:
        d = to_int(parts[0]);
        month = parts[1];
        break;
      default:
        return ({ -2 });
    }

    m = check_month(month);

    if (m<0)
      return ({ -4 });

    if (m==1)
    {
      if (d>29)
        return ({ -5 });
    }
    else
    {
      if (d>DAYS_PER_MONTH[m])
        return ({ -5 });
    }

    return ({m, d});
  }
}


/************************************************************************/
/*                                                                      */
/* void get_month_name(int month)                                       */
/*                                                                      */
/* Diese Funktion gibt den Namen eines Monats zurueck, der aus dem      */
/* Define MON_NAMES extrahiert wird.                                    */
/*                                                                      */
/************************************************************************/
string get_month_name(int month)
{
  if (!intp(month) || month<0 || month>11)
    return "";

  return capitalize(lower_case(efun::implode(efun::explode(MON_NAMES[month], " "), "")));
}


/************************************************************************/
/*                                                                      */
/* void get_date_str(mixed date)                                        */
/*                                                                      */
/* Diese Funktion gibt den kompletten String eines Datums zurueck.      */
/*                                                                      */
/************************************************************************/
string get_date_str(mixed date)
{
  int j;

  if (!date || !pointerp(date))
    return "";

  if ((j=sizeof(date))==2)
    return to_string(date[1])+"."+get_month_name(date[0]);
  else if (j==3)
    return to_string(date[2])+"."+get_month_name(date[1])+" "+date[0];

  return "";
}


/************************************************************************/
/*                                                                      */
/* string ListAllBirthdays()                                            */
/*                                                                      */
/* Diese Funktion wird von aussen aufgerufen und stellt einen String    */
/* mit allen Geburtstagen zusammen. Dabei werden nur die Daten, nicht   */
/* aber die Namen sortiert ausgegeben.                                  */
/*                                                                      */
/************************************************************************/
string ListAllBirthdays()
{
  int i, j, k, l, m, n, *month, *day, *date, big;
  string day_str, *names, text, text2;
  object player;

  month = sort_array(m_indices(birthdays), #'>);

  if (!(j=sizeof(month)))
    return "Es sind keine Geburtstage eingetragen.\n";

  player = this_player();
  date = get_date();
  big = 0;

  text = player->in_color(KAL1, "\nFolgende Tage sind als Geburtstage verzeichnet:\n"
         +"-----------------------------------------------\n");

  for (i=0;i<j;i++)
  {
    day = sort_array(m_indices(birthdays[month[i]]),#'>);
    day_str = "";

    for (l=sizeof(day),k=0;k<l;k++)
    {
      names = sort_array(birthdays[month[i]][day[k]],#'>);

      for (m=sizeof(names),n=0;n<m;n++)
        names[n] = capitalize(names[n]);

      if (!big && ((date[1]<=month[i] && date[2]<=day[k]) || date[1]<month[i]))
      {
        big = 1;
        text2 = player->in_color(KAL2, sprintf("%2d -->", day[k]));
      }
      else
        text2 = sprintf("%2d -->",day[k]);

      day_str += break_string(SOUL->CountUp(names), WD, "  "+text2+" ");
    }

    text += sprintf("%s\n%s", player->in_color(KAL1, get_month_name(month[i])+":"),
      day_str);
  }

  if (!big)
  {
    big = strstr(text," -->") - 3;
    text2 = text[0..big];
    text = text[(big+1)..];
    text = text2 + player->in_color(KAL2, text[0..5]) + text[6..];
  }

  return text;
}


/************************************************************************/
/*                                                                      */
/* string ListAllEntries()                                              */
/*                                                                      */
/* Diese Funktion wird von aussen aufgerufen und stellt einen String    */
/* mit allen Eintragungen zusammen. Dabei werden nur die Daten, nicht   */
/* aber die Eintragungen selber ausgegeben.                             */
/*                                                                      */
/************************************************************************/
string ListAllEntries()
{
  int *year, *month, *day, *date, i, j, k, l, m, n, o, p, big;
  string *names, text, mon_name;
  object player;

  year = sort_array(m_indices(entries), #'>);

  if (!(j=sizeof(year)))
    return "Es sind keine Eintraege vorhanden.\n";

  player = this_player();
  date = get_date();
  big = 0;

  text = player->in_color(KAL1, "\nFolgende Eintraege wurden gemacht:\n"
         +"------------------------------------\n");

  for (i-0;i<j;i++)
  {
    text += player->in_color(KAL1, sprintf("%d:\n", year[i]));
    month = sort_array(m_indices(entries[year[i]]), #'>);

    for (l=sizeof(month),k=0;k<l;k++)
    {
      mon_name = get_month_name(month[k]);
      day = sort_array(m_indices(entries[year[i]][month[k]]), #'>);

      for (n=sizeof(day),m=0;m<n;m++)
      {
        if (!big &&
            ((date[0]<=year[i] && date[1]<=month[k] && date[2]<=day[m]) ||
             (date[0]<=year[i] && date[1]<month[k]) ||
             (date[0]<year[i])))
        {
          big = 1;
          text += player->in_color(KAL2, sprintf("  %d.%s:\n", day[m], mon_name));
        }
        else
          text += player->in_color(KAL3, sprintf("  %d.%s:\n", day[m], mon_name));

        names = sort_array(m_indices(entries[year[i]][month[k]][day[m]]), #'>);

        for (p=sizeof(names),o=0;o<p;o++)
          names[o] = sprintf("%s: %d", capitalize(names[o]), 
            sizeof(entries[year[i]][month[k]][day[m]][names[o]]));

        text += break_string(SOUL->CountUp(names), WD, 4);
      }
    }
  }

  return text;
}


/***************************************************************/
/*                                                             */
/* int check_for_birthday(object player)                       */
/*                                                             */
/* Diese Funktion soll von aussen aufgerufen werden, sobald    */
/* ein sichtbarer Spieler das Wunderland betritt. Es wird      */
/* geprueft, ob der Spieler in der Liste ist und ob heute sein */
/* Geburtstag ist. Wenn ja, bekommen alle Spieler ausser ihm   */
/* selbst eine Meldung. Die Meldungen lassen sich ignorieren.  */
/*                                                             */
/*  ignoriere geburtstage                                      */
/*      - Ignoriert alle Meldungen                             */
/*  ignoriere geburtstag.<spielername>                         */
/*      - Ignoriert nur die Meldungen dieses Spielers          */
/*  ignoriere spieler                                          */
/*      - Ignoriert neben den meisten anderen Funktionen wie   */
/*        sagen, rufen und mitteilen auch die diese Meldung    */
/*                                                             */
/* Returnwerte:                                                */
/* neg : Fehler, nicht eingetragen oder kein Geburtstag        */
/*   1 : Geburtstag wurde ausgegeben                           */
/*                                                             */
/***************************************************************/
int check_for_birthday(object player)
{
  int i, j, d, m, *help, p1;
  string pname, ignore_all, ignore_player, *ignore;
  object *players;

  if (!player || !objectp(player))
    return -1;

  if (player->QueryProp(P_INVIS))
    return -2;

  pname = getuid(player);

  help = get_date();
  m = help[1];
  d = help[2];

  p1 = member(birthdays, m);
  p1 = (p1?member(birthdays[m], d):0);
  p1 = (p1?member(birthdays[m][d], pname):-1);

  if (p1<0)
    return -3;

  ignore_all = IGNORE_ALL;
  ignore_player = IGNORE_ALL[0..<2]+"."+pname;

  players = users() - ({player});
  pname=player->name(WER);

  for (j=sizeof(players),i=0;i<j;i++)
    if (!(ignore=players[i]->QueryProp(P_IGNORE)) || (pointerp(ignore) && 
        member(ignore,ignore_all)==-1 && member(ignore,ignore_player)==-1 &&
        member(ignore,pname)==-1))
    {
        tell_object(players[i],
          players[i]->in_color(KAL4, sprintf("%s hat heute Geburtstag.\n",
          capitalize(pname))));
    }

     tell_object(player, player->in_color(KAL4,
       sprintf("Herzlichen Glueckwunsch zum Geburtstag, %s.\n",
       capitalize(pname))));

  return 1;
}
