/*******************
** Eldarea MUDLib **
********************
**
** secure/news.c - mud news server
**
** CVS DATA
** $Date: 2000/11/30 16:10:52 $
** $Revision: 1.2 $
**
** CVS History
**
** $Log: news.c,v $
** Revision 1.2  2000/11/30 16:10:52  elatar
** header corrected
**
** Revision 1.1.1.1  1999/11/05 12:30:45  elatar
** Preparing mudlib for cvs control
**
**
*/

// Newsgroups:   string *groups, enthaelt die Namen aller Newsgroups.
//               string *groups[0] = namen;
//               string *groups[1] = savefiles;

// Group: array mit folgenden Elementen:
//               string name;      Gruppenname;
//               string owner;
//               string savefile;
//               string *deleters; Gruppe der Leute, die loeschen koennen
//               string *writers;  Wer kann in der Gruppe schreiben
//               string *readers;  Wer kann die Gruppe lesen ?
//               int dlevel;       Ab level>=dlevel kann jeder loeschen
//               int wlevel;       Ab level>=wlevel kann jeder schreiben
//               int rlevel;       Ab level>=rlevel kann jeder lesen
//               int expire;       -1 heisst "nie expiren"
//               int maxmessages;
//               mixed *messages;

// Message: array mit folgenden Elementen:
//               string group;
//           (*) string writer;
//           (*) string id;    Mudname+":"+time()
//           (*) int    time;
//               string title;
//               string message;
//               mixed * referenz; Array von Arrays, ({M_REF_ID,M_REF_WRITER,M_REF_TIME,M_REF_GROUP})
//                                 M_REF_ID 1=Reply (nur einmal), 2=Move (beliebig oft)
//   Die mit (*) gekennzeichneten Eintraege setzt der Demon selber

// Returnvalues fuer ALLE Funktionen:  
//          0 = Parameter error
//          1 = Ok.
//         -1 = no permission
//         -2 = no such group/group already existing

// name, owner, savefile koennen nur von Archwizards geaendert werden, der
// Rest auch vom Owner. Also koennen auch nur Arches neue Gruppen einrichten

#define INTERACTIVE  (this_interactive()?this_interactive():previous_object())
#define TIME_TO_CLEAR 1800 /* alle 30 Min */
#define MIN_TOUCHED 20

#include <wizlevels.h>
#include <language.h>
#include <config.h>
#include <news.h>

// #define DEBUG(x) if(find_player("holger")) tell_object(find_player("holger"), sprintf("%O\n", x))

#pragma strong_types

mixed *saveload; // Diese Variable dient Uebertragen von Daten in/aus savefiles

static mixed *grouplist;   // Groups und ihre save-Files, zudem LastTime
static mixed *newsgroups;  // Dies sind die Gruppen
static int lastwritten;    // letzter Artikel (Zeit)

varargs int expire_space(mixed num, int anzahl, int art_num);
static void save_group(mixed group);
static void save_group_list();
static int load_group(string name);
static int update_wtime(string name, int t);
int AskAllowedWrite(string group);
mixed GetBoardInfo(string board, int restricted);
varargs int GetNewsTime(string boardname);

void create() {
  int i;

  seteuid(getuid(this_object()));
  if(restore_object(NEWSPATH+"GroupList"))
    grouplist=order_alist(saveload);
  if(!pointerp(grouplist))
    grouplist=({({}),({}),});
  newsgroups=({({}),({}),});
  call_out("_clear_groups",TIME_TO_CLEAR);
  lastwritten=0;
  for(i=sizeof(grouplist[1]);i--;)
    if(lastwritten<to_int(grouplist[1][i][1]))
      lastwritten=to_int(grouplist[1][i][1]);
  call_out("expire_all",60);
}

int secure() {
  if(this_interactive() && IS_ARCH(this_interactive()))
    return 1;
  return 0;
}

/*
 * int AddGroup(string groupname, string owner);
 *  Funktion:
 *    Gruppe anlegen. (EM only!)
 *  Result:
 *    -3 = no owner
 *    -4 = savefile already in use
 */
int AddGroup(string name, string owner) {
  mixed *group;
  string savefile;
  string *savefilea;
  int i;

  if(!name || !owner) return 0;
  if(!secure()) 
    return -1;

  if (assoc(name, grouplist, -1) != -1) return -2; /* Gibt es schon */

  if (file_size("/"+SAVEPATH+owner[0..0]+"/"+owner+".o")<0) return -3;

  savefilea = explode(name,".");
  savefile = implode(savefilea,"/");
  if (file_size(NEWSPATH+savefile+".o")>=0) return -4;

  /* Notwendige Directories anlegen */
  for (i = 0; i < sizeof(savefilea)-1; i++) {
    mkdir(NEWSPATH+implode(savefilea[0..i],"/"));
  }

  group=({});

  group+=({name});
  group+=({owner});
  group+=({savefile});
  group+=({-1});
  group+=({({})});
  group+=({({})});
  group+=({({})});
  group+=({8});
  group+=({0});
  group+=({0});
  group+=({40});
  group+=({({})});
  group+=({1});
  newsgroups=insert_alist(name,group,newsgroups);
  save_group(group);
  grouplist=insert_alist(name,({savefile,0}),grouplist);
  save_group_list();
  return 1;
}

/*
 * int RemoveGroup(string groupname);
 *  Funktion:
 *    Gruppe loeschen.  (EM only!)
 */
int RemoveGroup(string name) {
  int num;

  if(!name) return 0;

  if(!secure()) 
    return -1;

  if(!pointerp(grouplist)) 
    return -2;

  if((num=assoc(name, grouplist[0]))==-1)
    return -2;

  catch(rm(NEWSPATH+grouplist[1][num][0]+".o"));
  grouplist=exclude_alist(num, grouplist);

  save_group_list();

  num=assoc(name,newsgroups[0]);
  if(num==-1) 
    newsgroups=exclude_alist(num,newsgroups);
  return 1;
}

/*
 * int SetGroup(string name, int dlevel, int wlevel, int rlevel, 
 *              int maxmessages, int expire);
 *  Funktion:
 *    Legt Loeschlevel (dlevel), Schreiblevel (wlevel), Leselevel (rlevel),
 *    die max. Anzahl Artikel und die Expire-Zeit fest. (EM & Owner only)
 */
int SetGroup(string name, int dlevel, int wlevel, int rlevel, 
	     int maxmessages, int expire) {
  int num;

  if ((num=load_group(name))==-1) return -2;

  if (newsgroups[1][num][G_OWNER]!=geteuid(INTERACTIVE) &&
      !IS_ARCH(INTERACTIVE)) return -1;

  newsgroups[1][num][G_DLEVEL]=dlevel;
  newsgroups[1][num][G_WLEVEL]=wlevel;
  newsgroups[1][num][G_RLEVEL]=rlevel;
  newsgroups[1][num][G_MAX_MSG]=maxmessages;
  newsgroups[1][num][G_EXPIRE]=expire;

  save_group(newsgroups[1][num]);
  return 1;
}

/*
 * AddAllowed(string name, mixed deleters, mixed writers, mixed readers);
 *  Funktion:
 *    Man kann ein Array oder String mit Namen von fuer die Gruppe 
 *    loesch/schreib/lese-Berechtigten fuer die Gruppe angeben.
 *    (EM & Owner only)
 */
int AddAllowed(string name, mixed deleters, mixed writers, mixed readers) {
  int num;

  if ((num=load_group(name))==-1) return -2;

  if (newsgroups[1][num][G_OWNER]!=geteuid(INTERACTIVE) &&
      !IS_ARCH(INTERACTIVE)) return -1;

  if (stringp(deleters)) deleters=({deleters});
  if (stringp(writers)) writers=({writers});
  if (stringp(readers)) readers=({readers});

  if (!deleters) deleters=({});
  if (!writers) writers=({});
  if (!readers) readers=({});

  newsgroups[1][num][G_DELETERS]+=deleters;
  newsgroups[1][num][G_WRITERS]+=writers;
  newsgroups[1][num][G_READERS]+=readers;

  save_group(newsgroups[1][num]);
  return 1;
}

/*
 * int RemoveAllowed(string name, mixed deleters, mixed writers, mixed readers);
 *  Funktion:
 *    Man kann ein Array oder String mit Namen von fuer die Gruppe 
 *    loesch/schreib/lese-Berechtigten fuer die Gruppe entfernen.
 *    (EM & Owner only)
 */
int RemoveAllowed(string name, mixed deleters, mixed writers, mixed readers) {
  int num;

  if ((num=load_group(name))==-1) return -2;

  if (newsgroups[1][num][G_OWNER]!=geteuid(INTERACTIVE) &&
      !IS_ARCH(INTERACTIVE)) return -1;

  if (stringp(deleters)) deleters=({deleters});
  if (stringp(writers)) writers=({writers});
  if (stringp(readers)) writers=({readers});

  if (!deleters) deleters=({});
  if (!writers) writers=({});
  if (!readers) readers=({});

  newsgroups[1][num][G_DELETERS]-=deleters;
  newsgroups[1][num][G_WRITERS]-=writers;
  newsgroups[1][num][G_READERS]-=readers;

  save_group(newsgroups[1][num]);
  return 1;
}

/*
 * int WriteNote(mixed message);
 *  Funktion:
 *    Schreibt einen Artikel. Format von message siehe Header!
 *  Result:
 *   -3 = Max number of msgs exceeded
 */
int WriteNote(mixed message) {
  int trust, num, ret;

  if(!pointerp(message)) return 0;

  if((num=load_group(message[M_BOARD]))==-1) return -2;

  trust=member(TRUSTED_OBJS,efun::explode(file_name(previous_object()),"#")[0]);
  ret=AskAllowedWrite(newsgroups[0][num]);
  if (trust!=-1) ret=1;

  if (ret==-1) return ret;
  
  if(ret==-3) {
    if(geteuid(INTERACTIVE)!=ROOTID) // ausser ROOT niemand
      return -3;
    else if(expire_space(num, 1)<1)  // Expire fuer ROOT (z.B. Merlin)
      return -3;
  }
  
  if (ret=1 && TRUSTED_OBJS[trust+1]==message[M_BOARD]) 
  {
    // Hier ist ein Objekt, was Artikel veroeffentlich darf, wie es will.
    // Wir lassen also den Artikel unangetastet. Insbesondere muss der
    // M_WRITER schon korrekt gesetzt sein. Ist immer nur fuer bestimmte
    // Rubriken freigegeben. M_TIME und M_ID werden hier gesetzt.
    /* tue nix */
  } 
  else 
  {/*
    if (lower_case(INTERACTIVE->name())==getuid(INTERACTIVE) ||
       (!query_once_interactive(INTERACTIVE) && INTERACTIVE->name()))
       // vis player oder automatischer NPC
      message[M_WRITER]=capitalize(INTERACTIVE->name(WER, 1));
    else // invis player*/
      message[M_WRITER]=capitalize(getuid(INTERACTIVE));
  }
  // Schauen, obs schon eine Nachricht mit diesem Timestamp gibt. Wenn
  // ja, ist die ID nicht eindeutig, der neue Artikel ist nicht von den
  // Newsclients ansprechbar so einfach. Deswegen einfach den Stamp so
  // lange in die Zukunft verlegen, bis ein 'freier' Platz gefunden
  // wurde. Damit es nicht so auffaellt, max 30 Sekunden Differenz
  // zulassen, sonst halt doch gleiche IDs
  ret=GetNewsTime(message[M_BOARD]);
  if (time()-ret <=0) {
    message[M_TIME]=ret;
    if (time()-ret >-30) message[M_TIME]++;
  } else message[M_TIME]=time();
  message[M_ID]=MUDNAME+":"+message[M_TIME];
  newsgroups[1][num][G_MESSAGES]+=({message});
  save_group(newsgroups[1][num]);
  update_wtime(message[M_BOARD], message[M_TIME]);
  return 1;
}

/*
 * int RemoveNote(string boardname, int notenummer);
 *  Funktion:
 *    Loescht Artikel aus der Rubrik
 *  Result:
 *   -3 = No such note
 */
int RemoveNote(string name, int note) {
  int num;
  int l;

  if (note<0) return 0;

  if ((num=load_group(name))==-1) return -2;

  l=sizeof(newsgroups[1][num][G_MESSAGES]);
  if (l<=note)
    return -3;

  if (newsgroups[1][num][G_OWNER] != geteuid(INTERACTIVE) &&
      !IS_ARCH(INTERACTIVE) && 
      newsgroups[1][num][G_DLEVEL]>query_wiz_level(INTERACTIVE) &&
      lower_case(newsgroups[1][num][G_MESSAGES][note][M_WRITER])
                             !=INTERACTIVE->query_real_name() &&
      member_array(geteuid(INTERACTIVE),newsgroups[1][num][G_DELETERS])==-1)
    return -1;

  if (l==1) 
    newsgroups[1][num][G_MESSAGES]=({});  
  else
    if (!note)
      newsgroups[1][num][G_MESSAGES]=newsgroups[1][num][G_MESSAGES][1..l-1];
    else 
      if (note==l-1) 
        newsgroups[1][num][G_MESSAGES]=newsgroups[1][num][G_MESSAGES][0..l-2];
      else
        newsgroups[1][num][G_MESSAGES]=newsgroups[1][num][G_MESSAGES][0..note-1]
                                   +newsgroups[1][num][G_MESSAGES][note+1..l-1];
  save_group(newsgroups[1][num]);
  return 1;
}

/*
 * mixed GetNotes(string boardname); 
 *  Funktion:
 *    gibt einen Array mit den Notes aus Rubrik 'boardname' zurueck.
 */
mixed GetNotes(string boardname) {
  int num;

  if((num=load_group(boardname))==-1) 
    return -2;

  if(newsgroups[1][num][G_OWNER] != geteuid(INTERACTIVE) &&
    !IS_ARCH(INTERACTIVE) && 
    newsgroups[1][num][G_RLEVEL]>query_wiz_level(INTERACTIVE) &&
    member_array(geteuid(INTERACTIVE),newsgroups[1][num][G_READERS])==-1)
      return -2; /* No such group for the requestor :) */
  return newsgroups[1][num][G_MESSAGES];
}

string GetStatistics() {
  string s;
  int i;

  s="Geladene Foren und wie oft sie seit dem letzten clear accessed wurden:\n";
  for (i=0;i<sizeof(newsgroups[0]);i++)
    s+=extract(newsgroups[0][i]+"               ",0,15)+" : "+
       newsgroups[1][i][sizeof(newsgroups[1][i])-1]+"\n";
  return s;
}

static void dump_file(string filename, mixed news) {
  int i;

  for (i=0;i<sizeof(news);i++)
    write_file(filename,news[i][M_TITLE]+" ("+news[i][M_WRITER]+", "+
      extract(dtime(news[i][M_TIME]),5,26)+"):\n"+
      news[i][M_MESSAGE]+
      "\n-----------------------------------------------"
      "------------------------------\n\n\n\n");
}

void expire(int num) {
  int to_expire,size,last;

  if(newsgroups[1][num][G_EXPIRE]<=0)
    return;
  to_expire=time()-newsgroups[1][num][G_EXPIRE];
  size=sizeof(newsgroups[1][num][G_MESSAGES]);
  last=size;
  if(!last)
    return;
  while(last && newsgroups[1][num][G_MESSAGES][last-1][M_TIME]>to_expire)
    last--;
  if(!last) 
    return;
  if(last==size) {
    dump_file("news/OLD."+newsgroups[0][num],newsgroups[1][num][G_MESSAGES]);
    newsgroups[1][num][G_MESSAGES]=({});
    save_group(newsgroups[1][num]);
    return;
  }
  dump_file("news/OLD."+newsgroups[0][num],
    newsgroups[1][num][G_MESSAGES][0..last-1]);
    newsgroups[1][num][G_MESSAGES]=newsgroups[1][num][G_MESSAGES][last..size-1];
    save_group(newsgroups[1][num]);
}

varargs int expire_space(mixed num, int anzahl, int art_num) {
  int last;
  // Gruppe 'num' ist voll. Nun werden hier Artikel rausgeschmissen,
  // die noch nicht das expire-Alter haben, hauptsache Platz :)
  // Ist anzahl != 0 so werden soviele Artikel ausgemustert.
  // Ist art_num != 0 so wird genau der Artikel mit dieser Nummer expired.
  // num ist entweder nummer (intern) oder rubrikname.
  if(stringp(num)) num=(int)load_group(num);
  if(!intp(num) || num<0) return -2;
  if(AskAllowedWrite(newsgroups[0][num])==-1)
    return -1;
  if(!art_num) {
    last= newsgroups[1][num][G_MAX_MSG];
    expire(num); // Erstmal normal versuchen
    if (sizeof(newsgroups[1][num][G_MESSAGES]) < last-(anzahl?anzahl-1:0))
      return 2;
    if (anzahl) {
      if (anzahl>last) /* nix */;
      else last=anzahl;
    } else last=last/10; // Auf 9/10 von MAX_MSG loeschen
    dump_file("news/OLD."+newsgroups[0][num],
                  newsgroups[1][num][G_MESSAGES][0..last-1]);
    newsgroups[1][num][G_MESSAGES]=newsgroups[1][num][G_MESSAGES][last..];
    save_group(newsgroups[1][num]);
    return 3;
  }
  last=sizeof(newsgroups[1][num][G_MESSAGES]);
  if (art_num>last || art_num<0) return 0;
  dump_file("news/OLD."+newsgroups[0][num],
                newsgroups[1][num][G_MESSAGES][art_num-1..art_num-1]);
  newsgroups[1][num][G_MESSAGES]=newsgroups[1][num][G_MESSAGES][0..art_num-2]+
    newsgroups[1][num][G_MESSAGES][art_num..];
  save_group(newsgroups[1][num]);
  return 1;
}

void expire_all() {
  int i;

  while(remove_call_out("expire_all")>=0);
  while(remove_call_out("expire")>=0);
  for(i=0;i<sizeof(newsgroups[1]);i++)
    call_out("expire",i*3,i);
    call_out("expire_all",86400);
}

void _clear_groups() {
  int i, todo, len;

  todo=1;
  if (!sizeof(newsgroups[1])) return;
  len=sizeof(newsgroups[1][0]);
  while(todo) {
    i=0;
    todo=0;
    while (i<sizeof(newsgroups[1])) {
      if (newsgroups[1][i][len-1]<MIN_TOUCHED) {
        exclude_alist(i,newsgroups);
        todo=1;
      }
      i++;
    }
  }
  for (i=0;i<sizeof(newsgroups[1]);i++)
    newsgroups[1][i][len-1]=0;
  call_out("_clear_groups",TIME_TO_CLEAR);
}

static void save_group(mixed group) {
  saveload=group[0..sizeof(group)-2]; /* Do NOT save the accessed-Info */
  save_object(NEWSPATH+group[G_SAVEFILE]);
}

static void save_group_list() {
  saveload=grouplist;
  save_object(NEWSPATH+"GroupList");
}

static int load_group(string name) {
  int num;

  num=assoc(name, newsgroups[0]);

  if(num!=-1) {
    newsgroups[1][num][sizeof(newsgroups[1][num])-1]++;
      return num;
  }

  num=assoc(name,grouplist[0]);

  if(num==-1) return -1;

  restore_object(NEWSPATH+grouplist[1][num][0]);
  if(pointerp(saveload) && !intp(saveload[G_EXPIRE]))
    saveload=saveload[0..G_EXPIRE-1]+({-1})+saveload[G_EXPIRE..<1];
  newsgroups=insert_alist(name,saveload+({1}),newsgroups);
  num=assoc(name,newsgroups[0]);
  expire(num);
  return num;
}

static int update_wtime(string name, int t) {
  int num;

  num=assoc(name,grouplist[0]);
  if(num<0) 
    return 0;
  lastwritten=t?t:time();
  grouplist[1][num][1]=lastwritten;
  save_group_list();
  return 1;
}

mixed GetGroups() { 
  mixed *returnlist;
  int i,num;

  if(IS_ARCH(INTERACTIVE))
    return sort_array(grouplist[0],#'>);

  returnlist=({});
  for(i=0; i<sizeof(grouplist[0]); i++) {
    num=load_group(to_string(grouplist[0][i]));
    if(newsgroups[1][num][G_RLEVEL]<=query_wiz_level(INTERACTIVE) ||
      newsgroups[1][num][G_OWNER]==geteuid(INTERACTIVE) ||
      member_array(geteuid(INTERACTIVE),newsgroups[1][num][G_READERS])!=-1)
        returnlist=returnlist+({grouplist[0][i]});
  }
  return sort_array(returnlist, #'>);
}
		    
/*
 * int AskAllowedWrite(string group); 
 *  Funktion:
 *    stellt fest, ob ein Player eine Note aufhaengen darf oder nicht
 */
int AskAllowedWrite(string group) {
  int num;

  if((num=(int)load_group(group))==-1)
    return -2;

  if(newsgroups[1][num][G_OWNER] != geteuid(INTERACTIVE) && 
    !IS_ARCH(INTERACTIVE) && geteuid(INTERACTIVE)!=ROOTID &&
    newsgroups[1][num][G_WLEVEL]>query_wiz_level(INTERACTIVE) &&
    member_array(geteuid(INTERACTIVE),newsgroups[1][num][G_WRITERS])==-1)
      return -1;
  
  if(sizeof(newsgroups[1][num][G_MESSAGES])>=newsgroups[1][num][G_MAX_MSG])
    return -3;
  return 1;
}

int AskAllowedRead(string group, object reader) {
  int num;
  if((num=load_group(group))==-1) return -2;
   if(newsgroups[1][num][G_OWNER] != geteuid(reader) &&
    !IS_ARCH(reader) &&
    newsgroups[1][num][G_RLEVEL]>query_wiz_level(reader) &&
    member_array(geteuid(reader),newsgroups[1][num][G_READERS])==-1)
      return -1;
  return 1;
}

string GetReference(int number, string group) {
  int num;
  mixed message;
  if((num=load_group(group))==-1) return "";   /* keine solche Gruppe */
  message=newsgroups[1][num][G_MESSAGES];
  if (sizeof(message) <= number) return "";
  message=message[number];
  if(sizeof(message))
    return message[M_WRITER]+":"+message[M_TIME]+":"+group;  /* Zuordnung */
  return "";
}

int GetNumber(string id, string board, string writer) {
  int num,size,i,pos;
  mixed message;
  pos = 0;
  if((num=load_group(board))==-1) return 0;
  size=sizeof(newsgroups[1][num][G_MESSAGES]);
  if (!size) return 0;
  for(i=0;i<size;i++) {
    message=newsgroups[1][num][G_MESSAGES][i];
    if(message[M_ID]==MUDNAME+":"+id &&
       message[M_WRITER]==writer) {
      pos = i + 1;
      i + size - 1;
    }
  }
  return pos;
}

/*
 * varargs int GetNewsTime(string boardname);
 *  Funktion:
 *    gibt zurueck, wann am entsprechenden Brett zum letzten Mal eine Note 
 *    befestigt wurde. Falls kein boardname angegeben wird, liefert 
 *    GetNewsTime() den Zeitpunkt, zu dem ueberhaupt eine neue Note 
 *    aufgehaengt wurde.
 */
varargs int GetNewsTime(string boardname) {
  int i, ltime;

  if(!boardname)
    return lastwritten;
  i=assoc(boardname,grouplist[0]);
  if(i==-1) 
    return -1;
  return grouplist[1][i][1];
}

/*
 * mixed GetBoardInfo(string board, int restricted);
 *  Funktion:
 *    Gibte Informationen ueber eine Rubrik zurueck.
 *  Result:
 *    mixed
 *     -2   - Rubrik existiert nicht.
 */
mixed GetBoardInfo(string board, int restricted) {
  int num;
  mixed ret;
  
  if((num=load_group(board))==-1) 
    return -2;
  
  if (restricted) {
    // Arrays alle kopieren!
    ret=newsgroups[1][num]+({});
    // Achtung hier Differenz im Format!
    ret[G_MESSAGES]=sizeof(ret[G_MESSAGES]);
    for (num=sizeof(ret); num--;) {
      if (pointerp(ret[num]) && sizeof(ret[num])) ret[num]+=({});
    }
    return ret;
  }

  if(newsgroups[1][num][G_OWNER]!=geteuid(INTERACTIVE) &&
    !IS_ARCH(INTERACTIVE)&& AskAllowedRead(board, INTERACTIVE)<1) 
      return -1;

  return newsgroups[1][num];
}



/* --- Wichtig! --- */

int query_prevent_shadow(object dummy) { return 1; }
