// This file is part of UNItopia Mudlib.
// ----------------------------------------------------------------
// File:	/secure/simul_efun/string.inc
// Description: Funktionen zum vereinfachten Stringhandling
// Author:	Freaky, Francis

#pragma strict_types
#pragma save_types

/***********************************************************************
 *
 * Stringfunktionen:
 *
 *	center(string, len, pad)
 *	copies(string, num)
 *	left(string, len, pad)
 *      liste(string_array [, verb]);
 *	right(string, len, pad)
 *	strip(string, option, pad)
 *	substr(string, n, len, pad)
 *	space(string, n, pad)
 *      wrap(str, len)
 *      wrap_say(str1, str2, [len1])
 *
 ***********************************************************************/

/*
FUNKTION: liste
DEKLARATION: string liste(string *str [, string pad])
BESCHREIBUNG:
Machst aus einem Stringarray eine normale Aufzaehlung:
    liste(({"bla","schnack","sabbel"})) -> 
	"bla, schnack und sabbel"
    liste() -> ""
    liste(({"bla","schnack","sabbel"}), " oder ") ->
	"bla, schnack oder sabbel"
VERWEISE: 
GRUPPEN: string, simul_efun
*/ 
varargs string liste(string *str, string pad) {
    if (!sizeof(str))
	return "";
    if (sizeof(str) == 1) 
	return str[0];
    return implode(str[0..<2], ", ")+(pad ? pad : " und " )+str[<1];
}

/*
FUNKTION: wrap
DEKLARATION: string wrap(string str [, int len (default: 75) ] )
BESCHREIBUNG:
wrap bricht einen String auf die Breite <len> um und schliesst ihn mit einem
'\n' ab. Wenn <len> nicht angegeben wird, wird auf 75 Zeichen ungebrochen.
GRUPPEN: string, simul_efun
VERWEISE: sprintf, copies, left, right, center, wrap_say
*/
varargs string wrap(string str, int len) {
    if (len<1)
	return sprintf("%-=75s\n",str);
    return sprintf("%-=*s\n",len,str);
}

/*
FUNKTION: wrap_say
DEKLARATION: string wrap_say(string str1, string str2 [, int len])
BESCHREIBUNG:
wrap_say() erzeugt einen String, der etwa so aussieht:
say(wrap_say("Monty sagt:","blablabla ....."));->

Monty sagt: blablabla blablabla blablabla blablabla blablabla blablabla 
	blablabla blablabla blablabla blablabla blablabla blablabla 
	blablabla blablabla ...

Der optionale Parameter bestimmt die Breite, auf die str2 umgebrochen
werden soll.
VERWEISE: sprintf, copies, left, right, center, wrap
GRUPPEN: string, simul_efun
*/
varargs string wrap_say(string str1, string str2, int len) { 
    if (len<1)
	len = 75;
    return sprintf("%s%=-*s\n", str1[0..7], len-8, str1[8..]+" "+str2);
}


/*
FUNKTION: copies
DEKLARATION: string copies(string pattern, int times)
BESCHREIBUNG:
Gibt 'times' aneinandergesetzte Kopien von 'pattern' zurueck.

Beispiele:
        copies("*",10)          ->      "**********"
        copies("hu",2)          ->      "huhu"
        copies("tra",0)         ->      ""

GRUPPEN: string, simul_efun
VERWEISE: sprintf, wrap, left, right, center
*/
string copies(string str,int len) {
    if (len <= 0 || str == "")
	return "";
    return sprintf("%'"+str+"'*s",strlen(""+str)*len,"");
}

/*
FUNKTION: center
DEKLARATION: string center(string text, int len [,string pattern] )
BESCHREIBUNG:
Gibt einen String der Laenge len zurueck, in dem text zentriert wurde.
Wenn noetig wird mit dem Zeichen pad aufgefuellt. Voreinstellung fuer
pad ist das Leerzeichen.

Beispiele:
        center("abc",7)                 ->      "  abc  "
        center("abc",8,"-")             ->      "--abc---"
        center("Der blaue Himmel",7)    ->      "laue Hi"

GRUPPEN: simul_efun, string
VERWEISE: sprintf, copies, left, right, wrap
*/
varargs string center(string text, int len, string pad) {
    int b;

    if (len<=0)
	return "";
    text = ""+text;
    if ((b=strlen(text))>=len) {
	b=(b-len)/2;
	return text[b..b+len-1];
	}
    b = (len-b) % 2;
    if (pad)
	return sprintf("%|*'"+pad+"'s",len+b,text)[b..len];
    return sprintf("%|*s",len+b,text)[b..len];
}

/*
FUNKTION: left
DEKLARATION: string left(string text, int len [,string pattern] )
BESCHREIBUNG:
Gibt einen String der Laenge len zurueck, in dem text linksbuendig
enthalten ist. Wenn noetig wird mit dem Zeichen pad aufgefuellt.
Voreinstellung fuer pad ist das Leerzeichen.

Beispiele:
        left("abc d",8)         -->     "abc d   "
        left("abc d",8,".")     -->     "abc d..."
        left("abc  def",7)      -->     "abc  de"

GRUPPEN: simul_efun, string
VERWEISE: sprintf, copies, center, right, wrap
*/
varargs string left(string text, int len, string pad) {
    if (len<=0)
	return "";
    if (pad)
	return sprintf("%'"+pad+"':-*s",len,""+text);
    return sprintf("%:-*s",len,""+text);
}

/*
FUNKTION: right
DEKLARATION: string right(string text, int len [,string pattern] )
BESCHREIBUNG:
Gibt einen String der Laenge len zurueck, in dem text rechtsbuendig
enthalten ist. Wenn noetig wird mit dem Zeichen pad aufgefuellt.
Voreinstellung fuer pad ist das Leerzeichen.

Beispiele:
        right("abc d",8)        -->     "   abc d"
        right("abc d",8,".")    -->     "...abc d"
        right("abc  def",7)     -->     "bc  def"

GRUPPEN: simul_efun, string
VERWEISE: sprintf, copies, center, right, wrap
*/
varargs string right(string text, int len, string pad) {
    if (len<=0)
	return "";
    text = ""+text; /* um eventuelle integerzahlen in string zu verwandeln */
    if (len<=strlen(text))
	return text[<len..];
    if (pad)
	return sprintf("%'"+pad+"':*s",len,text);
    return sprintf("%:*s",len,text);
}

/*
FUNKTION: substr
DEKLARATION: string substr(string text, int n, int len [,string pattern] )
BESCHREIBUNG:
Gibt den Teil von text zurueck, der mit dem n'ten Zeichen beginnt
und len lang ist. Wenn noetig wird mit dem Zeichen pad aufgefuellt.
Voreinstellung fuer pad ist das Leerzeichen.
Wird len nicht angegeben, so wird der Rest von text zurueckgegeben.

Beispiele:
        substr("abc",2)         -->     "bc"
        substr("abc",2,4)       -->     "bc  "
        substr("abc",2,6,".")   -->     "bc...."

GRUPPEN: simul_efun, string
VERWEISE: sprintf, copies, center, right, wrap
*/
varargs string substr(string text, int n, int len, string pad)
{
#if 0
   if (!stringp(text) || !intp(n) || !intp(len) || (pad && !stringp(pad)))
	if (find_player("freaky"))
	    tell_object(find_player("freaky"),sprintf(
	    "SUBSTR: text:%O n:%O len:%O pad:%O PO:%O\n",text,n,len,pad,
	    previous_object()));
#endif

    if (len > 0)
	return sprintf("%'"+(pad||" ")+"'-*.*s",
                  len, len, to_string(text)[(--n<0?0:n)..]);
    if (!len)
        return to_string(text)[(--n<0?0:n)..];
}

#if 0
/*
 dummy FUNKTION: strip
DEKLARATION: string strip(string text [,string opt [,string char]] )
BESCHREIBUNG:
Entfernt vorangestellte (opt = "l"), nachfolgende (opt = "t") oder
vorangestellte und nachfolgende (opt = "b") Zeichen aus text.
Voreinstellung fuer opt ist "b".
Das zu entfernende Zeichen wird in char angegeben. Voreinstellung
hierfuer ist das Leerzeichen.

Beispiele:
        strip("   abc   ")              -->     "abc"
        strip("   abc   ","l")          -->     "abc   "
        strip("   abc   ","t")          -->     "   abc"
        strip("00012.700","","0")       -->     "12.7"
        strip("00012.700","","10")      -->     "2.7"

GRUPPEN: simul_efun, string
VERWEISE: sprintf, copies, center, right, wrap
*/
varargs string strip(string text, string opt, string char) {
    int textlen, begin, end, o;
    string ptab;

    if (!text || text == "")
	return "";

    if (!strlen(char))
	ptab=" \t";
    else
	ptab=char;

    if (!opt)
	o='b';
    else
	o=lower_case(opt)[0];

    textlen = strlen(text);

    if (o == 'b' || o == 'l') {
	while(member(ptab,text[begin])>=0 && begin<textlen)
	    begin++;
	if (begin==textlen)
	    return "";
	}

    end=textlen-1;
    if (o == 'b' || o == 't')
	while(member(ptab,text[end])>=0)
	    end--;

    return text[begin..end];
}
#endif

/*
FUNKTION: strip
DEKLARATION: string strip(string text)
BESCHREIBUNG:
Entfernt vorangestellte und nachfolgende Spaces und Tabs aus text.

Beispiele:
        strip("   abc   ")              -->     "abc"

GRUPPEN: simul_efun, string
VERWEISE: sprintf, copies, center, right, wrap
*/
string strip(string text) {
    int begin, end;

    if ((end = strlen(text)-1) < 0)
	return "";

    while(text[begin]==' ' || text[begin]=='\t')
	begin++;

    if (begin > end)
	return "";

    while(text[end]==' ' || text[end]=='\t')
	end--;

    return text[begin..end];
}

/*
FUNKTION: space
DEKLARATION: string space(string str [,int n [,string pad]] )
BESCHREIBUNG:
Loescht alle ueberschuessigen Spaces, und Tabs zwischen den Woertern
und fuellt die Zwischenraeume durch den String <pad> auf

Beispiele:
        space("   hallo  wie gehts   ") -> "hallo wie gehts"
        space("hallo  wie gehts",2)     -> "hallo  wie  gehts"
        space("hallo  wie gehts",0,"")  -> "hallowiegehts"
        space("hallo  wie gehts",3,"+") -> "hallo+++wie+++gehts"

GRUPPEN: simul_efun, string
VERWEISE: 
*/
varargs string space(string str, int n, string pad) {
    if (n<=0)
	n=1;
    if (!pad)
	return implode(explode(implode(explode(""+str,"\t")," ")," ")-({""}),sprintf("%*s",n,""));
    return implode(explode(implode(explode(""+str,"\t")," ")," ")-({""}),copies(pad,n));
}


/*
FUNKTION: add_dot_to_msg
DEKLARATION: string add_dot_to_msg(string msg)
BESCHREIBUNG:
Haengt an <msg> einen Punkt an, wenn <msg> kein Satzzeichen !?.,;:
am Ende enthaelt.
GRUPPEN: simul_efun, string
VERWEISE: 
*/
string add_dot_to_msg(string msg)
{
   int c;
   if(stringp(msg) && 
      (c = msg[<1]) &&
      member(({'.','!','?',',',':',';'}), c) < 0)
      return msg + ".";
   return msg;
}
