/*******************
** Eldarea MUDLib **
********************
**
** filename - short desc
**
** CVS DATA
** $Date: 1999/11/05 12:30:46 $
** $Revision: 1.1.1.1 $
**
** longdesc
**
** CVS History
**
** $Log: deklin.c,v $
** Revision 1.1.1.1  1999/11/05 12:30:46  elatar
** Preparing mudlib for cvs control
**
**
*/
// This file is part of UNItopia Mudlib.
// ----------------------------------------------------------------
// File:	/secure/simul_efun/deklin.c
// Description:	Deklination von Pronomina, Substantiven, Adjektiven
// Author:	Garthan (15.01.93)
// Modified by: Kurdel (05.02.97) string_parser fuer Argumentlisten erweitert

#pragma save_types
#pragma strict_types

/* =================== Defines && Includes ===================== */

#include "/sys/deklin.h"  /* ART_..., FALL_... und OBJ_... Konstanten */
#include "/sys/invis.h"
#include "/sys/error.h"

#undef  DEKLIN_LOG
#undef  DEKLIN_DEBUG

#define PRON_DER    0
#define PRON_BEST   1
#define PRON_UNBEST 2
#define PRON_1_PERS 3
#define PRON_2_PERS 4
#define PRON_3_PERS 5

#define QUERY(x,p) (mappingp(p) ? funcall(p[x]) \
				: (mixed)call_other(p,"query_"+(x)))

/* =================== Global Data ============================= */

private static mixed *pronomen =  // Alle Pronomen
({
   ({ // PRON_DER
      ({ "das", "des", "dem", "das" }),
      ({ "der", "des", "dem", "den" }),
      ({ "die", "der", "der", "die" }),
      ({ "die", "der", "den", "die" }),
   }),
   ({ // PRON_BEST
      ({ "es","es", "em", "es" }),
      ({ "er","es", "em", "en" }),
      ({ "e", "er", "er", "e"  }),
      ({ "e", "er", "en", "e"  }),
   }),
   ({ // PRON_UNBEST
      ({ "" , "es", "em", ""   }),
      ({ "" , "es", "em", "en" }),
      ({ "e", "er", "er", "e"  }),
      ({ "e", "er", "en", "e"  }),
   }),
   ({ // PRON_1_PERS
      ({  "ich", "meiner", "mir", "mich" }),
      ({  "ich", "meiner", "mir", "mich" }),
      ({  "ich", "meiner", "mir", "mich" }),
      ({  "wir", "unser" , "uns", "uns"  })
   }),
   ({ // PRON_2_PERS
      ({  "du" , "deiner", "dir", "dich" }),
      ({  "du" , "deiner", "dir", "dich" }),
      ({  "du" , "deiner", "dir", "dich" }),
      ({  "ihr", "euer"  , "euch","euch" })
   }),
   ({ // PRON_3_PERS
      ({  "es",  "seiner", "ihm",  "es"  }),
      ({  "er",  "seiner", "ihm",  "ihn" }),
      ({  "sie", "ihrer" , "ihr",  "sie" }),
      ({  "sie", "ihrer" , "ihnen","sie" })
   })
});

private static mixed *endungen =  // Alle Adjektivendungen
({
   // bestimmt          | unbestimmt     | ohne Pronom     |
   // ------------------+----------------+-----------------+
   // der, dieser       | ein, mein, dein|                 |
   //                   | sein, kein     | Plural von ein  |
   // ------------------+----------------+-----------------+ 
   //    Nom,Gen,Dat,Akk| Nom,Gen,Dat,Akk| Nom,Gen,Dat,Akk |
   ({ 0 ,"" ,"n","n","" , "s","n","n","s", "s","n","m","s" }), // Sg: neutrum 
   ({ 0 ,"" ,"n","n","n", "r","n","n","n", "r","n","m","n" }), // Sg: maskulin
   ({ 0 ,"" ,"n","n","" , "" ,"n","n","" , "" ,"r","r",""  }), // Sg: feminin
   ({ 0 ,"n","n","n","n", "n","n","n","n", "" ,"r","n",""  })  // Pl: alle drei
});

private static string * genders = ({ "maennlich", "weiblich" });
private static string * jemand  = ({ "", "ens", "em", "en" });

private static string dativ_plural_endungen = "ans";
private static string genitiv_endungen = "szx";

/* ============== Generic Utility Functions ==================== */

/*
FUNKTION: not_alone
DEKLARATION: int not_alone(object ob)
BESCHREIBUNG:
Liefert 1 zurueck, wenn sich weitere Objekte mit derselben Id wie ob
in der Umgebung von ob befinden.
(wird beispielsweise zur Automatischen Atikel Auswahl (ART_AAA) in
der Grammatik/Deklination verwendet.
GRUPPEN: simul_efun, grammatik
*/
int not_alone(mixed ob)
{
   string *ids;

   if(!objectp(ob))
      return 0;
   if(!(ids = (string *)ob->query_id()) || !sizeof(ids) || !environment(ob))
      return 0;
   return present(ids[0]+" 2", environment(ob)) != 0;
}


/*
FUNKTION: auto_owner_search
DEKLARATION: object auto_owner_search(mixed ob)
BESCHREIBUNG:
Liefert das naechst-umgebende lebende Objekt von ob zurueck
oder 0 falls kein lebendes Objekt gefunden.

Mit anderen Worten: Es wird der Besitzer eines Objekts
zurueckgeliefert, sprich derjenige, der es mit sich rumtraegt.
VERWEISE: sein, seines, seinem, seinen, ihr, ihres, ihrem, ihren
GRUPPEN: simul_efun, grammatik
*/
object auto_owner_search(mixed ob)
{
   do
   {
      ob = mappingp(ob) ?
	 ob["environment"] :
	 environment(ob);
   }
   while(ob && !(objectp(ob) && living(ob)));
   return ob;
}

/* ============== STRING_PARSER FUNCTIONS ====================== */

/*
FUNKTION: string_parser
DEKLARATION: varargs mixed *string_parser(string str, int tp_flag, int use_tp)
BESCHREIBUNG:
Der uebergebene String str wird nach Funktionen durchsucht und ein Array
aus Closures und Strings aufgebaut und zurueckgegeben, welches fuer die
Konstruktion einer lambda-Closure verwendet werden kann.
Enthaelt der String str Funktionen, so muessen sie in der Form "$fun(args)"
geschrieben werden. Dabei ist fun der Name der Funktion und args ist eine
Liste von durch Kommas getrennten Argumenten.
Argumente koennen Mappings, Arrays, Strings, Integerzahlen, Closures oder
Symbole sein.
Die Strings werden ohne "" geschrieben, Zahlen werden als Argumente und
Mappingeintraege automatisch als solche erkannt.
Will man partout eine Zahl als Stringargument, muss man sie in "" setzen,
das Quoten mit \ in LPC-Strings nicht vergessen!
Ist tp_flag gesetzt, wird Funktionen ohne Argument entweder 'tp (use_tp==1)
oder OBJ_TP uebergeben.

string_parser wird von mixed_to_closure verwendet, im Normalfall kommt man
damit vor allem bei msgs von v_items und Bewegungsmeldungen in Beruehrung.
Argumente sind vor allem praktisch, wenn man sein/ihr verwenden moechte
bei der Erwaehnung von Sinnen oder Koerperteilen des Spielers in den msgs,
man muss dann nicht mehr eigenhaendig Closures konstruieren.

Beispiele:

   Bewegungsmeldungen:
   
      "$Der() naehert sich $dir()."
      "$Der(OBJ_TP,besoffen) torkelt $dir() davon."
      "Du stolperst ueber $den(OBJ_TP,\"\")." (kein Adjektiv)
      
   v_item-msgs:
   
      "$Des(OBJ_TP,ungeschickt) Versuch schlaegt fehl."
      "$Sein(([name:haare,gender:saechlich,plural:1]),({lila,lilan}),
         OBJ_TP) stehen $ihm() gut."
      "$Ihr(([name:augen,plural:1,gender:saechlich]),mued,OBJ_TP,
         erschoepft) fallen zu."
      "$Ihr(([name:nase,gender:weiblich]),empfindlich,OBJ_TP
       ,herumschnueffelnd) kraeuselt sich."

VERWEISE: mixed_to_closure
GRUPPEN: grundlegendes
*/
// Sieht zwar lang aus, letzlich teilt sichs in Fallunterscheidungen auf,
// in einem String ist meist eh nur eine Funktion, selten zwei, und
// auch meist parameterlos.
// Ist dann die Closure konstruiert, lebt sichs voellig ungeniert ...
varargs mixed *string_parser(string str, int tp_flag, int use_tp)
{
   string *parts, fun, *s, *e, element;
   int i,j,k,l, pos;
   mixed head, op, params;
   mapping m;

   if (!str)
      return 0;
   if (str=="") {
      head=str;
      return head;
   }
   parts = regexplode(str,"\\([^\\(\\)]*\\)")-({",",""});
   for (i = 0; i < sizeof(parts); i++)
   {
      fun = 0;
      element = parts[i]; // spart staendige Array-Indizierungen
      if (sscanf(element, "%s$%s", str, fun) && fun != "" &&
	 (fun == "dir" || fun == "Dir" || function_exists(
         (fun[<1]=='(' ? fun[0..<2] : fun), this_object())==__FILE__[0..<3]))
         // aus dem touch wird dann noch ein this_object()
      {
         if (str!="")
            if (head)
               head = ({ #'+, head, str });
            else
               head = str;
	 switch(fun)
	 {
	    case "dir" :
	       op = 'richtung; i++; break;
	    case "Dir" :
	       op = ({ #'capitalize, 'richtung }); i++; break;
	    default:
               if ((pos = strstr(fun, "(")) > -1)
               {
                  parts = parts[0..i-1] +
                    ( (pos == strlen(fun) - 1) ? ({ fun[0..<2] }) :
                    ({ fun[0..pos-1], fun[pos+1..<1] }) )
                    + parts[i+1..<1];
                  element = parts[++i];
                  fun = fun[0..pos-1];
                  if (i<sizeof(parts))
                  {
                     params = ({element});  // der 1. Parameter von fun
                     i++;
                     while (i < sizeof(parts) && (element = parts[i])[0] != ')'
                        && (element[0] == '(' || element[0] == ','))
                        // Parameterliste vervollstaendigen
                     {
                        if (element[0] == '(') // weiterer Parameter von fun
                           params += ({element});
                        else
                        {
                           if ((j = strstr(element,")")) != -1)
                           {  // Klammer schliesst Parameterliste ab
                              params += ({element[1..j-1]});
                              parts[i] = element[j..];
                              i--; // den Reststring bearbeiten wir nochmal
                           }
                           else
                              params += ({element[1..]}); // Param. nach Komma
                        }
                        i++;
                     }
                     if (i<sizeof(parts)) // nach schliessender Klammer weiter
                        parts[i] = element[strstr(element,")") + 1..];
                     i--;
                  }
                  else
                  {
                     params = ({""}); // kein Parameter ( Klammer zu fehlt)
                     break;
                  }
               }
               else
               {  // hier haben wir nur einen Parameter ...
                  if (++i<sizeof(parts))
                     params = ({parts[i]});
                  else
                     params = ({""}); // oder keinen (keine Klammern gesetzt)
               }
               // Jetzt das Parameterfeld bearbeiten:
               for (j = 0; j < sizeof(params); j++)
               {
                  element = params[j]; // Arrayindizierungen sparen
                  if (element && strlen(element) > 1)
                  {
                     if (element[0..1] == "({") // ein Array also
                     {
                        params[j] = quote(explode(element[2..<3],","));
                        continue; // zum naechsten Parameter gehen
                     }
                     if (element[0..1] == "([") // ein Mapping
                     {
                        m = ([]);
                        for (k = sizeof(s = explode(element[2..<3], ",")); k--;)
                        {
                          e = explode(s[k],":");
                          if (to_string(l = to_int(e[1])) == e[1])
                             m += ([e[0]:l]);
                          else
                             m += ([e[0]:e[1]]);
                        }
                        params[j] = m;
                        continue; // zum naechsten Parameter gehen
                     }
                     if (element[0] == '(' && element[<1] == ')')
                     {
                        element = params[j] = element[1..<2]; // () loeschen
                     }
                  }
                  if (to_string(k = to_int(element)) == element)
                  {  // Integerparameter
                     params[j] = k;
                     continue;
                  }
                  if (element[0] == '"' && element[<1] == '"')
                  {
                     element = params[j] = element[1..<2]; // evtl. "" loeschen
                     if (element == "")
                       continue;
                  }
                  if (sizeof(s = (explode(element,",") -({""}))) > 1)
                  {
                     // durch Komma getrennte Parameterlisten aufloesen
                     params = params[0..j-1] + s + params[j+1..<1];
                     j--; // aufgeloeste Parameterliste hier weiterbearbeiten
                  }
                  else {
                        element = sizeof(s) ? s[0] : 0;
                     switch(element)
               	     {
               		  case "OBJ_TO":
               		     params[j] = previous_object(); break;
	               	  case "OBJ_PO":
	               	     params[j] = OBJ_PO; break;
	               	  case "OBJ_TP":
	               	     if(use_tp)
	               		params[j] = 'tp;
	               	     else
	               		params[j] = OBJ_TP; break;
	               	  case "OBJ_OW":
	               	     params[j] = OBJ_OW; break;
	               	  case "OBJ_TI":
	               	     params[j] = OBJ_TI; break;
	               	  default:
	               	     if(stringp(element) && element != "")
	               	     {
		               	if (element[0..1] == "#'")
			           params[j] = ({#'call_other, 
			              previous_object(), element[2..]});
			        else if (element[0..0] == "'")
			           params[j] = quote(element[1..]);
		             }
		             else if (tp_flag) 
			        if (use_tp)
			           params[j] = 'tp;
			        else
			           params[j] = OBJ_TP;
		             else 
			        params[j] = previous_object();
		     }
                  }
              }
            op = ({ symbol_function(fun) }) + params; // Puh ...
         }
      }
      else
         op = element;
      if (head)
         head = ({ #'+, head, op }); // hintendrauf summieren
      else
         head = op; // Tja, hier faengts eben an
   }
   return head;
}

#define MTC_IS_STR(a,b) (stringp(a)?string_parser(a,b):a)

/*
FUNKTION: mixed_to_closure
DEKLARATION: varargs closure mixed_to_closure(mixed mix, mixed *symbols, int tp_flag)
BESCHREIBUNG:
Dokumentation siehe: /doc/funktionsweisen/messages
VERWEISE: closure_to_string, string_parser, convert_message
GRUPPEN: grundlegendes
*/

varargs closure mixed_to_closure(mixed mix, mixed *symbols, int tp_flag)
{
   int a;
   mixed ret;

   if(stringp(mix))
      ret = string_parser(mix, tp_flag);
   else if(pointerp(mix))
      for(a = sizeof(mix)-2, ret = MTC_IS_STR(mix[<1], tp_flag);
          a >= 0; a--)
         ret = ({ #'+, MTC_IS_STR(mix[a],tp_flag), ret });
   else if(closurep(mix))
      return mix;
   else
      ret = "";

   return unbound_lambda(symbols?symbols:({}), ret);
}

/*
FUNKTION: convert_message
DEKLARATION: mixed convert_message(mixed str)
BESCHREIBUNG:
Dokumentation siehe: /doc/funktionsweisen/messages
Konvertiert eine alte Message in eine neue in dem sie 
$Der(OBJ_TP) vorne anhaengt, falls kein $ in str enthalten.
Liefert die konvertierte Meldung zurueck.
VERWEISE: mixed_to_closure, string_parser, convert_message
GRUPPEN: grundlegendes
*/

mixed convert_message(mixed str)
{
   if(stringp(str) && strstr(str,"$")<0)
      return "$Der(OBJ_TP) " + str;
   return str;
}

/* ============== END OF STRING_PARSER FUNCTIONS ====================== */

private string deklin_err(int err, mixed who, int art, int fall, string msg)
{
#ifdef DEKLIN_LOG
   write_file("/log/DEKLIN",
      (this_player() ? (string)this_player()->query_name() : " --- ")+": "+
      "err #"+err+" by "+file_name(previous_object())+" "+
      (this_player() ? (string)this_player()->query_command() : query_verb())+
      sprintf("\n   who: %O, art: %O, fall: %O\n", who, art, fall));
#endif
   do_error2(msg+sprintf("\n\t(who:%O)\n", who),
   previous_object()?file_name(previous_object()):0,
   (who && objectp(who))?file_name(who):0,0);

   return this_interactive() && this_interactive()->query_wiz_level() ?
	  "'DeklinErr "+err+"'" : "irgend etwas";
}

// Funktion zur Rekonstruktion der urspruenglich aufgerufenen Funktion :(
// Nur fuer's Errhandling relevant.
private string deklin_function(int art, int fall)
{
   switch(art)
   {
      case ART_EIN:
	 switch(fall)
	 {
	    case FALL_GEN: return "eines()";
	    case FALL_DAT: return "einem()";
	    case FALL_AKK: return "einen()";
	    case FALL_NOM: 
	    default:       return "ein()";
	 }
      case ART_DER:
	 switch(fall)
	 {
	    case FALL_GEN: return "des()";
	    case FALL_DAT: return "dem()";
	    case FALL_AKK: return "den()";
	    case FALL_NOM: 
	    default:       return "der()";
	 }
      case ART_DIESER:
	 switch(fall)
	 {
	    case FALL_GEN: return "dieses()";
	    case FALL_DAT: return "diesem()";
	    case FALL_AKK: return "diesen()";
	    case FALL_NOM: 
	    default:       return "dieser()";
	 }
      case ART_DEIN:
	 switch(fall)
	 {
	    case FALL_GEN: return "deines()";
	    case FALL_DAT: return "deinem()";
	    case FALL_AKK: return "deinen()";
	    case FALL_NOM: 
	    default:       return "dein()";
	 }
      case ART_SEIN:
	 switch(fall)
	 {
	    case FALL_GEN: return "seines()";
	    case FALL_DAT: return "seinem()";
	    case FALL_AKK: return "seinen()";
	    case FALL_NOM: 
	    default:       return "sein()";
	 }
      case ART_ER:
	 switch(fall)
	 {
	    case FALL_GEN: return "wessen()";
	    case FALL_DAT: return "ihm()";
	    case FALL_AKK: return "ihn()";
	    case FALL_NOM: 
	    default:       return "er()";
	 }
      default:
	 switch(fall)
	 {
	    case FALL_GEN: return "wessen()";
	    case FALL_DAT: return "wem()";
	    case FALL_AKK: return "wen()";
	    case FALL_NOM: 
	    default:       return "wer()";
	 }
   }
}


/* ============== Grammar Utility Functions ==================== */


   /* -------- Artikel/Pronomen ----------- */

/*
FUNKTION: query_pronom
DEKLARATION: varargs string query_pronom(mixed who, int art, int fall, mixed owner)
BESCHREIBUNG:
Diese Funktion liefert Artikel und Pronomen aller Art.

  who   Das Objekt dessen Pronomen oder Artikel gewuenscht ist.

	ODER eine virtuelles Objekt in der Form:
	   ([ "name" : "...", "gender" : "..." ]) 
	oder bei Plural
	   ([ "name" : "...", "gender" : "...", "plural": 1 ]) 

  art
     ART_EIN        unbest. Artikel    (ein)
     ART_DER        bestimmter Artikel (der)
     ART_DIESER     demonstrativ       (dieser)
     ART_MEIN       possesiv 1.Person  (mein)
     ART_DEIN       possesiv 2.Person  (dein)
     ART_SEIN       possesiv 3.Person  (sein)  Besitzerangabe moeglich
     ART_ICH        personal 1.Person  (ich)
     ART_DU         personal 2.Person  (du)
     ART_ER         personal 3.Person  (er)
     ART_KEIN       negiert            (kein)
     ART_JENER      \
     ART_MANCHER     > Selbsterklaerend :)
     ART_WELCHER    /

  fall
     FALL_DEF   0   Nominativ
     FALL_NOM   1   Nominativ
     FALL_GEN   2   Genitiv
     FALL_DAT   3   Dativ
     FALL_AKK   4   Akkusativ

  owner Besitzer des Objekts. Falls 0 wird es aus dem environment berechnet.
        (nur fuer art == ART_SEIN bzw. ART_NUR_SEIN verwendet)

Die Konstanten ART_..., FALL_... werden in deklin.h definiert.

Alle Parameter sind optional.
Defaultwerte sind dann
   who:      this_object()
   art:      0  (bestimmter Artikel)
   fall:     0  (Nominativ)
   owner:    0  (auto owner search: living environment object)

Die Verwendung von der, dem, den,... wird empfohlen!!! Siehe dort.
VERWEISE: query_deklin, wer, der
GRUPPEN: simul_efun, grammatik
*/

varargs string query_pronom(mixed who, int art, int fall, mixed owner)
{
   string gender_string;
   int plural, gender;

   if(!who)
      who = previous_object();
   if(objectp(who) || mappingp(who))
   {
      gender_string = QUERY("gender", who);
      plural = QUERY("plural", who);
   }
   else
      return 0;

   if(!fall)
      fall = FALL_NOM;
   fall--;

   gender = plural ? 3 : 1 + member_array(gender_string, genders);

   switch(art)
   {
      case ART_EIN:
         return "ein"+pronomen[PRON_UNBEST][gender][fall];
      case ART_DER:
         return pronomen[PRON_DER][gender][fall];
      case ART_SEIN:
	 return (QUERY("gender",owner) == "weiblich" ? "ihr" : "sein") +
	        pronomen[PRON_UNBEST][gender][fall];
      case ART_MEIN:
         return "mein"+pronomen[PRON_UNBEST][gender][fall];
      case ART_KEIN:
         return "kein"+pronomen[PRON_UNBEST][gender][fall];
      case ART_DEIN:
         return "dein"+pronomen[PRON_UNBEST][gender][fall];
      case ART_ICH:
         return pronomen[PRON_1_PERS][gender][fall];
      case ART_DU:
         return pronomen[PRON_2_PERS][gender][fall];
      case ART_ER:
         return pronomen[PRON_3_PERS][gender][fall];
      case ART_DIESER:
         return "dies"+pronomen[PRON_BEST][gender][fall];
      case ART_JENER:
         return "jen"+pronomen[PRON_BEST][gender][fall];
      case ART_MANCHER:
         return "manch"+pronomen[PRON_BEST][gender][fall];
      case ART_WELCHER:
         return "welch"+pronomen[PRON_BEST][gender][fall];
   }
}


   /* -------- Adjektive ----------- */


/*
FUNKTION: query_deklin_ein_adjektiv
DEKLARATION: varargs string query_deklin_ein_adjektiv(mixed adj, int fall, string gender_string, int plural)
BESCHREIBUNG:
Diese Funktion liefert ein dekliniertes Adjektiv.

adj
   hat die Syntax eines ELEMENTS der Adjektivliste:
      string    Grundform eines Adjektivs                "gruen", "blau",...
         ODER
      string *  Ein zweielementiges Feld:                ({"lila", "lilan"})
                ({ grundform, unregelmaessiger_wortstamm })

fall  (default 1)
      1            Nominativ (mit bestimmtem Artikel davor)
      2            Genitiv        "          "
      3            Dativ          "          "
      4            Akkusativ      "          "
      5            Nominativ (mit unbestimmtem Artikel davor)
      6            Genitiv        "          "
      7            Dativ          "          "
      8            Akkusativ      "          "
      9            Nominativ (ohne Artikel davor)
     10            Genitiv        "          "
     11            Dativ          "          "
     12            Akkusativ      "          "

gender_string   (default saechlich)
      "saechlich"
      "maennlich"
      "weiblich"

plural   (default 0)
      0            Singularform
      1            Pluralform

An sich braucht man diese Funktion nie direkt, da man alles auch mit
den Funltionen wer,wessen,wem,wen und dessen Spezialformen erreichen kann.

BEISPIEL:
   query_deklin_ein_adjektiv("gross", FALL_DAT, "maennlich")   --> "grossen"

VERWEISE: set_adjektiv, query_adjektiv, adjektiv,
          add_adjektiv, delete_adjektiv,
          query_deklin_adjektiv, query_deklin, wer, der
GRUPPEN: simul_efun, grammatik
*/

varargs string query_deklin_ein_adjektiv(mixed adj, int fall, string gender_string, int plural)
{
   string stamm;
   int gender;


   if(stringp(adj))
      stamm = adj;
   else if(pointerp(adj) && sizeof(adj) > 1 && stringp(adj[1]))
      stamm = adj[1];
   else
      return 0;

   if(!fall)
      fall = FALL_NOM;

   gender = plural ? 3 : 1 + member_array(gender_string, genders);

   return stamm+"e"+endungen[gender][fall];
}


/*
FUNKTION: query_deklin_adjektiv
DEKLARATION: varargs string query_deklin_adjektiv(object who, mixed adj, int fall)
BESCHREIBUNG:
Diese Funktion liefert eine fertig deklinierte (gebeugte), komma
getrennte Liste von Adjektiven eines Objekts oder ein fremdes,
vorher uebergegebenes, dekliniertes Adjektiv.

who
   Das Objekt dessen Adjektive dekliniert werden.

adj
   hat ENTWEDER die Syntax eines ELEMENTS der Adjektivliste:
      string    Grundform eines Adjektivs  ("gruen", "blau",...)
      string *  Ein zweielementiges Feld:
                ({ grundform, unregelmaessiger_wortstamm })
   ODER folgende Syntax zur AUSWAHL bestimmter Adjektive des Objekts
      int i :  i == 0 : ALLE Adjektive des Objekts
               i >  0 : die ERSTEN i Adjektive des Objekts
               i <  0 : das -i. Adjektiv des Objekts
      int *ip: ip == ({ a, b }) Alle Adjektive vom a. bis zum b.
               Ist a, b negativ oder 0 dann wird von hinten gezaehlt.
               ( 1 ist das erste Adjektiv,
                 0 ist das letzte Adjektiv,
                -1 das vorletzte,...)

fall  0,1   Nominativ
      2     Genitiv
      3     Dativ
      4     Akkusativ
      5     Nominativ unbestimmte Form
      6     Genitiv unbestimmte Form
      7     Dativ  unbestimmte Form
      8     Akkusativ unbestimmte Form

BEISPIEL:
   Das Objekt ob habe folgende Adjektivliste mit set_adjektiv definiert
   und sei maennlich.

   set_adjektiv( ({"offen", ({ "lila", "lilan" }), "voll"}) );

   Dann liefert         (wenn zweites Argument fehlt, dann Nominativ)
      query_deklin_adjektiv(ob)               "offene, lilane, volle"
      query_deklin_adjektiv(ob,0)             "offene, lilane, volle"
      query_deklin_adjektiv(ob,1)             "offene"
      query_deklin_adjektiv(ob,2)             "offene, lilane"
      query_deklin_adjektiv(ob,3,4)           "offenen, lilanen, vollen"
      query_deklin_adjektiv(ob,-1)            "offene"
      query_deklin_adjektiv(ob,-2,4)          "lilanen"
      query_deklin_adjektiv(ob,-3)            "vollen"
      query_deklin_adjektiv(ob,({1,2})        "offene, lilane"
      query_deklin_adjektiv(ob,({2,3})        "lilane, volle"
      query_deklin_adjektiv(ob,({2,2})        "lilane"
      query_deklin_adjektiv(ob,({2,0})        "lilane, volle"
      query_deklin_adjektiv(ob,({2,-1})       "lilane"

      query_deklin_adjektiv("rote",3)      "roten"
      query_deklin_adjektiv(({"rosa","rosan"}),3) "rosanen"

VERWEISE: set_adjektiv, query_adjektiv, adjektiv,
          add_adjektiv, delete_adjektiv,
          set_id, query_id, id, add_id, delete_id,
          query_deklin, wer, der
GRUPPEN: simul_efun, grammatik
*/

varargs string query_deklin_adjektiv(mixed who, mixed adj, int fall)
{
   int i, s, end;
   string res;
   mixed * adjektiv;

   string gender;
   int plural;


   if(!who)
      who = previous_object();
   if(objectp(who) || mappingp(who))
   {
      gender = QUERY("gender", who);
      plural = QUERY("plural", who);
   }
   else
      return 0;


   /* Adjektiv explizit angegeben? */

   if(stringp(adj) || pointerp(adj) && stringp(adj[1]))
      return query_deklin_ein_adjektiv(adj, fall, gender, plural);

   /* Ab hier Adjektive des Objekts */

   if(!(adjektiv = QUERY("adjektiv", who)))
      return "";

   /* Ab hier Adjektivauswahl */

   i = 0;
   end = 0;
   s = sizeof(adjektiv);

   if(intp(adj))
   {
      if(adj < 0)    /* Ein bestimmtes Adjektiv gewaehlt? */
         return -adj <= sizeof(adjektiv) ?
            query_deklin_ein_adjektiv(adjektiv[-adj-1], fall, gender, plural) :
            0;

                     /* Alle Adjektive von 0 bis adj-1 */
      end = ( !adj ? s : (adj > s ? s : adj) )-1;
   }
   else if(pointerp(adj) && sizeof(adj) == 2 && intp(adj[0]) && intp(adj[1]))
   {
      /* Bereich gewaehlt */
      i = adj[0]-1;
      end = adj[1]-1;
   }

   if(s)
   {
      i   = (i   + s) % s;
      end = (end + s) % s;
   }

   res = "";
   for(; i <= end; i++)
   {
      res += query_deklin_ein_adjektiv(adjektiv[i], fall, gender, plural);
      if(i != end)
         res += ", ";
   }
   return res;
}



   /* -------- Substantiv/Hauptwort/Nomen ----- */


/*
FUNKTION: get_genitiv
DEKLARATION: string get_genitiv(string name)
BESCHREIBUNG:
Liefert die Genitiv-Form eines (Personal)-Namens zurueck.
Bsp: Francis  -> Francis'
     Detlef   -> Detlefs
VERWEISE: ihr, ihres, ihrem, ihren
GRUPPEN: simul_efun, grammatik
*/

string get_genitiv(string name)
{
   if(name && name != "")
   {
      if(member(genitiv_endungen, name[<1]) >= 0)
         return name+"'";
      return name+"s";
   }
   return name;
}


/*
FUNKTION: query_deklin_name
DEKLARATION: string query_deklin_name(mixed who, int fall)
BESCHREIBUNG:
Diese Funktion liefert ein dekliniertes Nomen.

who       Das Objekt dessen Namen gewuenscht ist

fall
      1   Nominativ
      2   Genitiv
      3   Dativ
      4   Akkusativ


Kein Genitiv: Die Funktion liefert einfach who->query_cap_name()
Genitiv     : Die Funktion haengt bei maennlichen Formen
              "es", "s", oder "'" an, je nach Endbuchstaben des
              Namens und who->query_personal().

VERWEISE: query_deklin, wer, der
GRUPPEN: simul_efun, grammatik
*/

string query_deklin_name(mixed who, int fall)
{
   string name, anfang, ende;
   int special, pos;

   if(!(name = mappingp(who) ?
        ((name = funcall(who["name"])) ? capitalize(name) : 0) :
        (string)who->query_cap_name()))
      return 0;
   if ((pos = strstr(name, " ")) != -1 || (pos=strstr(name, ",")) != -1)
   {
      anfang = name[0 .. pos - 1];
      ende = name[pos .. <1];
   }
   else {
      anfang = name;
      ende = "";
   }
   switch(fall)
   {
      case FALL_NOM:
	 return name;
      case FALL_DAT:
	 if(QUERY("plural", who) && member(dativ_plural_endungen, anfang[<1]) < 0)
	    return anfang + "n" + ende;
         // fallthru
      case FALL_AKK:
	 if(anfang[<1] == 'e' && anfang[<2] != 'e' &&
	    !QUERY("plural", who) && 
	    !QUERY("personal", who) &&
	    QUERY("gender", who) == "maennlich")
	    return anfang + "n" + ende;
	 return name;
      case FALL_GEN:
	 if(QUERY("gender", who) == "weiblich" || QUERY("plural", who))
	    return name;
	 if(anfang[<1] == 'e' && anfang[<2] != 'e' &&
            !QUERY("plural", who) && 
            !QUERY("personal", who) &&
            QUERY("gender", who) == "maennlich")
            name = anfang + "n" + ende;
         else
	    special = member(genitiv_endungen, anfang[<1]) >= 0;
	 if(QUERY("personal", who))
	    return anfang + (special ? "'" : "s") + ende;
	 return anfang + (special ? "es": "s") + ende;
   }
}




/* ================= deklin kernel functions =================== */

/* parse_deklin_object expandiert Abkuerzungen fuer who und owner
 * in query_deklin und query_deklin_owner Aufrufen.
 */

private mixed parse_deklin_object(mixed who)
{
   if(intp(who))
      if(who < 0)
         who = previous_object(1-who);
      else switch(who)
      {
         case OBJ_TO : /* 0 */
            who = previous_object();  break;
         case OBJ_PO : /* 1 */
            who = previous_object(1); break;
         case OBJ_TP : /* 2 */
            who = this_player();      break;
         case OBJ_OW : /* 3 */
            who = auto_owner_search(previous_object());
            break;
         case OBJ_TI : /* 4 */
            who = this_interactive(); break;
         default:
            who = previous_object();
      }
   return who;
}


/*
FUNKTION: query_deklin
DEKLARATION: varargs string query_deklin(mixed who, int art, int fall, mixed adjektiv, mixed owner)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens. Dabei wird beruecksichtigt,
ob das Objekt ein personal_name hat (query_personal()==1).

  who   Das Objekt dessen deklinierte Form gewuenscht ist.

	ODER eine virtuelles Objekt in der Form:
	   ([ "name" : "...", "gender" : "..." ]) 
	oder bei Plural
	   ([ "name" : "...", "gender" : "...", "plural": 1 ]) 

        ODER eine Zahl:
           OBJ_TO  0   this_object()  (default)
           OBJ_PO  1   previous_object()
           OBJ_TP  2   this_player()
           OBJ_OW  3   auto_owner_search  (sonst this_player())
           OBJ_TI  4   this_interactive()


  art
     ART_EIN            unbest. Artikel    (ein) + Adjektive + Nomen
     ART_DER            bestimmter Artikel (der) + Adjektive + Nomen
     ART_DIESER         demonstrativ       (dieser)  + Adjektive + Nomen
     ART_MEIN           possesiv 1.Person  (mein) + Adjektive + Nomen
     ART_DEIN           possesiv 2.Person  (dein) + Adjektive + Nomen
     ART_SEIN           possesiv 3.Person  (sein) + Adjektive + Nomen
     ART_ICH            personal 1.Person  (ich)
     ART_DU             personal 2.Person  (du)
     ART_ER             personal 3.Person  (er)
     ART_KEIN           negiert            (kein)
     ART_JENER      \
     ART_MANCHER     > Selbsterklaerend :)
     ART_WELCHER    /

     ART_KEINS          Kein Artikel, nur die Adjektive und das Nomen.
     ART_KEINS_BEST     Kein Artikel, nur die Adjektive und das Nomen.
     ART_KEINS_UNBEST   Kein Artikel, nur Adjketive(unbestimmte Form) und
                        das Nomen.
     ART_NUR_DER        nur bestimmter Artikel   (keine Adj., kein Nomen)
			(ART_DER | ART_NO_ADJEKTIV | ART_NO_NOMEN)
     ART_NUR_EIN        nur unbest. Artikel      (keine Adj., kein Nomen)
			(ART_EIN | ART_NO_ADJEKTIV | ART_NO_NOMEN)
     ART_NUR_DEIN       nur Possesivpronomen 2.P.(keine Adj., kein Nomen)
     ART_NUR_SEIN       nur Possesivpronomen 3.P.(keine Adj., kein Nomen)
     ART_NUR_DIESER     nur demonstr. Artikel (keine Adj., kein Nomen)
     ART_NUR_KEIN       (ART_KEIN | ART_NO_ADJEKTIV | ART_NO_NOMEN)

     ART_AAA            Automatische Artikelauswahl (ein/der) + Adj + Nomen
     ART_NUR_AAA        Nur Artikel nach automatischer Auswahl (k.Adj.,k.Nomen)
                        (Je nach Anzahl gleicher Objekte in der Umgebung wird
                         ein oder der benutzt.)


  fall
     FALL_DEF   0   Nominativ
     FALL_NOM   1   Nominativ
     FALL_GEN   2   Genitiv
     FALL_DAT   3   Dativ
     FALL_AKK   4   Akkusativ


  adjektiv  (mixed)
     string  ->  Grundform eines Adjektivs
                 ""         KEINE Adjektive
     string* ->  ({ unregelm.Grundform, Deklinationsstamm }) eines Adjektivs
     int i   ->  i == 0   ALLE Adjektive des Objekts who.
                 i >  0   Die ersten i Adjektive des Objekts who.
                 i <  0   Das -i. te Adjektiv des Objekts who.
     int*    ->  ({ a, b }) Die a bis b. Adjektive des Objekts who.
                 ({})       KEINE Adjektive


  owner Besitzer des Objekts. Falls 0 wird es aus dem environment berechnet.
        (nur fuer art == ART_SEIN bzw. ART_NUR_SEIN verwendet)

Die Konstanten ART_..., FALL_..., OBJ_... werden in deklin.h definiert.

Mit Hilfe von adjektiv koennen alle oder bestimmte Adjektive des Objekts
oder ein fremdes Adjektiv in den Ergebnisstring zwischen Artikel/Pronom und
Substantiv gestellt werden.
(Adjektive eines Objekts setzt man mit set_adjektiv, siehe dort.)

Alle Parameter sind optional.
Defaultwerte sind dann
   who:      this_object()
   art:      0  (bestimmter Artikel)
   fall:     0  (Nominativ)
   adjektiv: 0  (alle Adjektive des Objekts who)
   owner:    0  (auto owner search: living environment object)

BEISPIELE:
   who sei eine Person namens Anton
     query_deklin(who);                   liefert: "Anton"
   who sei ein Ork
     query_deklin(who,ART_EIN,3,"boesen");      liefert: "einem boesen Ork"
   what sei ein Auto und who eine weibliche Person
     query_deklin(what,ART_SEIN,4,0,who);        liefert: "ihr Auto"


query_deklin verwendet:
   auto_owner_search
   parse_deklin_object

   query_pronom
   query_deklin_adjektiv
   query_deklin_name

   query_personal
   query_plural
   query_eigen

Die Verwendung von der, dem, den, ... anstelle von query_deklin
wird empfohlen!!! Siehe dort.
VERWEISE: wer, der, query_pronom, set_adjektiv
GRUPPEN: simul_efun, grammatik
*/

varargs string query_deklin(mixed who, int art, int fall, mixed adjektiv, mixed owner)
{
   string satzteil, adj, pronom, title;
   mixed name, virt_env;
   int is_pers_pron, is_pers_name, only_pron;
   int flags, oart, append_personal;

#ifdef DEKLIN_DEBUG
   printf("query_deklin:: who:%O, art:%O, fall:%O, adjektiv:%O, owner:%O\n",
          who, art, fall, adjektiv, owner);
#endif
   if(!intp(art))
      return deklin_err(DEKLIN_ERR_ILLEGAL_ART,
	 who, art, fall, 
	 "Bad type of argument 'art' to "+deklin_function(art, fall)+
	 " (must be int)");
   if(objectp(adjektiv))
      return deklin_err(DEKLIN_ERR_ILLEGAL_ADJ,
	 who, art, fall, 
	 "Bad argument 'adjektiv' to "+deklin_function(art, fall)+
	 " (is object, must be adj)");

   /*  ==NEWSFLASH:   D O N ' T     P A N I C ! ! !== */

   /* --Objektumwandlungen-- */

   who = parse_deklin_object(who); /* Abkuerzungen expandieren */

   if(owner)  /* Wenn nicht auto_owner, dann Abkuerzungen expandieren */
      owner = parse_deklin_object(owner);

   if(!objectp(who) && !mappingp(who))
      return deklin_err(DEKLIN_ERR_ILLEGAL_WHO,
	 who, art, fall, 
	 "Bad argument 'who' to "+deklin_function(art, fall)+
	 " (must be object or virtual object)");

   /* --Fallumwandlungen-- */

   if(!fall)                   /* Defaultfall: Nominativ */
      fall = FALL_NOM;

   /* --Artumwandlungen-- */

   if(mappingp(who) &&
      (mappingp(virt_env = who["environment"]) || 
       virt_env && !(objectp(virt_env) && living(virt_env))))
   {
      oart = art;
      art = ART_DER;
      if(append_personal = (objectp(virt_env) && 
	 virt_env->query_personal() &&
	 !sizeof((mixed*)virt_env->query_adjektiv())))
	 art |= ART_NO_PRONOM;
   }

   flags = art;
   art &= ART_MASK;
   if(!art)
      art = ART_DER;


   if((flags & ART_INVIS ||
      !(flags & ART_VIS) && (QUERY("invis", who) & V_ATOM_INVIS)) 
         &&
      !(art & ART_NO_NOMEN))  // Bei art: ***_NUR_*** nicht jemand setzen
	 return objectp(who) && living(who) ?          
	    "jemand"+jemand[fall-1] :
	    "etwas";

   if((art == ART_SEIN || art == ART_DEIN || art == ART_MEIN) && 
      !owner && !(owner = auto_owner_search(who)))
      flags |= ART_AUTO;

   if(flags & ART_AUTO)   /* Automatische Artikel Auswahl */
      art = not_alone(who) ? ART_EIN : ART_DER;

   if(art == ART_EIN && 
      (QUERY("eigen",who) || QUERY("personal",who)))
      art = ART_DER;

   if(art == ART_EIN && QUERY("plural", who))
   {
      art = ART_BLANK;
      flags |= ART_NO_PRONOM;
   }

   /* --Artumwandlungen enden hier-- */

   only_pron = flags & ART_NO_ADJEKTIV && 
	       flags & ART_NO_NOMEN &&
               !(flags & ART_NO_PRONOM);

   is_pers_pron =         /* Ist ein Personalpronomen gewuenscht? */
      art == ART_ER || art == ART_DU || art == ART_ICH;     /* (er/du) */


   /* --Adjektivlogik-- */

   adj = is_pers_pron ||
	 only_pron || 
         stringp(adjektiv) && adjektiv == "" ||
         pointerp(adjektiv) && !sizeof(adjektiv) ||
	 flags & ART_NO_ADJEKTIV ?
      0 :
      query_deklin_adjektiv(who, 
			    adjektiv,
                            art == ART_BLANK ?
			      fall + 8 :
			      (art == ART_DER || art == ART_DIESER ?
				 fall :
				 fall + 4));
   adj = adj == "" ? 0 : adj;

   /* --Nomenlogik-- */

   if(!(flags & ART_NO_NOMEN))   /* NAME holen */
   {
      if(!(name = query_deklin_name(who, fall)))
	 return deklin_err(DEKLIN_ERR_MISSING_NAME,
	    who, art, fall,
	    "Object passed to "+deklin_function(art, fall)+
	    " has no name");
      if((is_pers_name = QUERY("personal", who)) &&
	 (title = QUERY("personal_title", who)))
      {
	 name = title + " " + name;
	 adj = 0;
      }
   }

   /* --Artikel/Pronomenlogik-- */


   /* Soll ein Pronom/Artikel vorangestellt werden?
      1  Personalnames bekommen keine Artikel,    Garthan
         (es sei denn, sie haben Adjektive)       der deklinierte Garthan
         wohl aber alle anderen Pronomina.        dieser Garthan
      2  wenn explizit kein Artikel/Pronom
         gewuenscht, dann gibt's auch keinen      bunter Hund
    */
   if((!is_pers_name || adj || !(art == ART_DER || art == ART_EIN)) && // 1 
       !(flags & ART_NO_PRONOM)                                        // 2 
         ||
       only_pron)
      /* Pronom/Artikel nur holen wenn benoetigt */
      pronom = query_pronom(who, art, fall, owner);


   /*  ==NEWSFLASH:   K E E P   C O O L  , it's nearly over.== */

   /* --Satzteil Zusammenbau-- */

   satzteil = "";
   if(pronom)
      satzteil += pronom;
   if(!is_pers_pron)
   {
      if(pronom && (name || adj))
         satzteil += " ";
      if(adj)
      {
         satzteil += adj;
         if(name)
            satzteil += " ";
      }
      if(name)
         satzteil += name;
   }

   /* Eventuell rekursiv ergaenzen */
   if(oart)
      if(append_personal)
	 return get_genitiv((string)virt_env->query_cap_name())+" "+ satzteil;
      else if(who["prep"])
	 return satzteil+" "+who["prep"]+" "+ 
		query_deklin(virt_env, oart, FALL_DAT);
      else
	 return satzteil+" "+
		query_deklin(virt_env, oart, FALL_GEN);
   else
      return satzteil;

   /*  ==NEWSFLASH:   T H A T ' S  I T   , you've got the message? == */
}



/*
 * query_deklin_owner macht die Dreckarbeit fuer die Funktionen 
 * ihr, ihren, ihrem.
 * Funktioniert aehnlich wie query_deklin. Frag Garthan...
 */

varargs string query_deklin_owner(
   mixed who,   int art, int fall, mixed adjektiv,
   mixed owner, int oart, mixed oadjektiv, string name)
{
   string owner_part, res;
   mixed virt_env;

   who = parse_deklin_object(who); /* Aus eventuellen Abk. Objekt machen */
   if(!objectp(who) && !objectp(owner) && !mappingp(who))
      return deklin_err(DEKLIN_ERR_2_AUTO_OWNER_WITH_VIRT_OBJ,
	 who, art, fall,
	 "Call to ihr/es/em/en() with virtual object and without explicit owner"
	 );
   if(!owner)
      owner = auto_owner_search(who);
   if(!owner)  /* ihr ohne Besitzer */
      return query_deklin(who, ART_AAA, fall, adjektiv);

   owner = parse_deklin_object(owner); /* Aus eventuellen Abk. Objekt machen */

   if(mappingp(who) &&
      (mappingp(virt_env = who["environment"]) ||
       virt_env && !(objectp(virt_env) && living(virt_env))))
      if(who["prep"])
	 return query_deklin(who, art, fall, adjektiv)+" "+
		who["prep"]+" "+
		query_deklin_owner(virt_env, ART_DER, FALL_DAT,0,
				   mappingp(virt_env) ?
				      virt_env["environment"]:
				      environment(virt_env),
				   oart, FALL_GEN, oadjektiv);
      else
	 return query_deklin(who, art, fall, adjektiv)+" "+
		query_deklin_owner(virt_env, ART_DER, FALL_GEN,0,
				   mappingp(virt_env) ?
				      virt_env["environment"]:
				      environment(virt_env),
				   oart, FALL_GEN, oadjektiv);

   owner_part = query_deklin(owner, oart, FALL_GEN, oadjektiv);

   if(objectp(owner) &&
      owner->query_personal() &&
      // Sehr heuristisch, aber schneller...
      //	 eigentlich sollte man hier testen, ob kein artikel/adjektiv
      //	 da ist...
      (owner->query_personal_title() ||
      strlen(owner_part) > 0 && sizeof(explode(owner_part," ")) == 1)
      )
   {
      if((string)owner->query_gender() == "weiblich" &&
	 member(genitiv_endungen, owner_part[<1]) < 0)
	 owner_part += "s";
      return 
	 owner_part
	 +" "+
	 (res = query_deklin(who, ART_KEINS, fall, adjektiv))+
	 (objectp(who) || mappingp(who) ? "" : (res==""?"":" ")+name);
   }
   else                                          /* unpersoenlicher Besitzer */
      return
	 (objectp(who) || mappingp(who) ?
	     query_deklin(who, art,              fall, adjektiv):
	     query_deklin(who, art|ART_NO_NOMEN, fall, adjektiv)
		+" "+name)
	 +" "+
	 owner_part;
}


/* ===== Zwischenschicht zwischen query_deklin und user ======== */


/*
FUNKTION: wer
DEKLARATION: varargs string wer(mixed who, int art, mixed adjektiv, object owner)
BESCHREIBUNG:
Liefert im Nominativ deklinierte Form eines Objektnamens.
Dabei wird beruecksichtigt, ob das Objekt ein personal_name hat
(query_personal()==1).

  who   Das Objekt dessen deklinierte Form im Nominativ gewuenscht ist.

	ODER eine virtuelles Objekt in der Form:
	   ([ "name" : "...", "gender" : "..." ]) 
	oder bei Plural
	   ([ "name" : "...", "gender" : "...", "plural": 1 ]) 

        ODER eine Zahl:
           OBJ_TO  0   this_object()  (default)
           OBJ_PO  1   previous_object()
           OBJ_TP  2   this_player()
           OBJ_OW  3   auto_owner_search  (sonst this_player())
           OBJ_TI  4   this_interactive()

  art
     ART_EIN            unbest. Artikel    (ein) + Adjektive + Nomen
     ART_DER            bestimmter Artikel (der) + Adjektive + Nomen
     ART_DIESER         demonstrativ       (dieser)  + Adjektive + Nomen
     ART_MEIN           possesiv 1.Person  (mein) + Adjektive + Nomen
     ART_DEIN           possesiv 2.Person  (dein) + Adjektive + Nomen
     ART_SEIN           possesiv 3.Person  (sein) + Adjektive + Nomen
     ART_ICH            personal 1.Person  (ich)
     ART_DU             personal 2.Person  (du)
     ART_ER             personal 3.Person  (er)
     ART_KEIN           negiert            (kein)
     ART_JENER      \
     ART_MANCHER     > Selbsterklaerend :)
     ART_WELCHER    /

     ART_KEINS          Kein Artikel, nur die Adjektive und das Nomen.
     ART_KEINS_BEST     Kein Artikel, nur die Adjektive und das Nomen.
     ART_KEINS_UNBEST   Kein Artikel, nur Adjketive(unbestimmte Form) und
                        das Nomen.
     ART_NUR_EIN        nur unbest. Artikel      (keine Adj., kein Nomen)
     ART_NUR_DER        nur bestimmter Artikel   (keine Adj., kein Nomen)
     ART_NUR_DIESER     nur demonstr. Artikel (keine Adj., kein Nomen)
     ART_NUR_MEIN       nur Possesivpronomen 1.P.(keine Adj., kein Nomen)
     ART_NUR_DEIN       nur Possesivpronomen 2.P.(keine Adj., kein Nomen)
     ART_NUR_SEIN       nur Possesivpronomen 3.P.(keine Adj., kein Nomen)
     ART_NUR_KEIN       nur Negationspronomen kein (keine Adj., kein Nomen)

     ART_AAA            Automatische Artikelauswahl (ein/der) + Adj + Nomen
     ART_NUR_AAA        Nur Artikel nach automatischer Auswahl (k.Adj.,k.Nomen)
                        (Je nach Anzahl gleicher Objekte in der Umgebung wird
                         ein oder der benutzt.)

  adjektiv  (mixed)
     string  ->  Grundform eines Adjektivs
                 ""         KEINE Adjektive
     string* ->  ({ unregelm.Grundform, Deklinationsstamm }) eines Adjektivs
     int i   ->  i == 0     ALLE Adjektive des Objekts who.
                 i >  0     Die ersten i Adjektive des Objekts who.
                 i <  0     Das -i. te Adjektiv des Objekts who.
     int*    ->  ({ a, b }) Die a bis b. Adjektive des Objekts who.
                 ({})       KEINE Adjektive

  owner Besitzer des Objekts. Falls 0 wird es aus dem environment berechnet.
        (nur fuer art == ART_SEIN bzw ART_NUR_SEIN verwendet)

Die Konstanten ART_..., FALL_..., OBJ_... werden in deklin.h definiert.

Mit Hilfe von adjektiv koennen alle oder bestimmte Adjektive des Objekts
oder ein fremdes Adjektiv in den Ergebnisstring zwischen Artikel/Pronom und
Substantiv gestellt werden.
(Adjektive eines Objekts setzt man mit set_adjektiv, siehe dort.)

Alle Parameter sind optional.
Defaultwerte sind dann
   who:      this_object()
   art:      0  (bestimmter Artikel)
   adjektiv: 0  (alle Adjektive des Objekts who)
   owner:    0  (auto owner search: living environment object)


Beispiele fuer wer, wem, wen
   who  sei eine Person namens Anton
   ork  sei ein Ork ohne Namen

     Wer(who)+" zieht "+wem(ork,ART_EIN)+" einen Pullover an."
        liefert:
     "Anton zieht einem Ork einen Pullover an."

     Wer(ork,ART_DIESER,"boese")+
         " laesst sich das von "+wem(who)+" nicht gefallen."
        liefert:
     "Dieser boese Ork laesst sich das von Anton nicht gefallen."

     Wer(ork)+" haette "+wen(who,ART_ER)+" am liebsten davongejagt."
        liefert:
     "Der Ork haette ihn am liebsten davongejagt."

     Wer(ork)+" patscht "+wem(who)+" auf "+
     wen((["name"  :"finger",
	   "gender":"maennlich",
	   "plural":1]), ART_SEIN, 0, who)+"."
        liefert:
     "Der Ork patscht Anton auf seine Finger."


wer, wem, wen sind Spezialfaelle von query_deklin.

ANSTELLE VON wer, wen, wem SOLLTEN DIE BEFEHLE:
  der, dem, den, ein, einem, einen, dein, deinem, deinen,
  sein, seinem, seinen, dieser, diesem, diesen, er, ihm, ihn
  Der, Dem, Den, Ein, ...
BENUTZT WERDEN!!!!!  (siehe dort)
VERWEISE: der, query_pronom, query_deklin, set_adjektiv
GRUPPEN: simul_efun, grammatik
*/
varargs string wer(mixed who, int art, mixed adjektiv, object owner)
{
   return query_deklin(who, art, FALL_NOM, adjektiv, owner);
}
varargs string Wer(mixed who, int art, mixed adjektiv, object owner)
{
   string res;
   return (res = query_deklin(who, art, FALL_NOM, adjektiv, owner)) ?
      capitalize(res) : 0;
}

/*
FUNKTION: wessen
DEKLARATION: varargs string wessen(mixed who, int art, mixed adjektiv, object owner)
BESCHREIBUNG:
Liefert im Genitiv deklinierte Form eines Objektnamens.
Dabei wird beruecksichtigt, ob das Objekt ein personal_name hat.
Weitere Beschreibung ->wer
VERWEISE: wer, wen, wem
GRUPPEN: simul_efun, grammatik
*/
varargs string wessen(mixed who, int art, mixed adjektiv, object owner)
{
   return query_deklin(who, art, FALL_GEN, adjektiv, owner);
}
varargs string Wessen(mixed who, int art, mixed adjektiv, object owner)
{
   string res;
   return (res = query_deklin(who, art, FALL_GEN, adjektiv, owner)) ?
      capitalize(res) : 0;
}


/*
FUNKTION: wem
DEKLARATION: varargs string wem(mixed who, int art, mixed adjektiv, object owner)
BESCHREIBUNG:
Liefert im Dativ deklinierte Form eines Objektnamens.
Dabei wird beruecksichtigt, ob das Objekt ein personal_name hat.
Weitere Beschreibung ->wer
VERWEISE: wer, wen
GRUPPEN: simul_efun, grammatik
*/
varargs string wem(mixed who, int art, mixed adjektiv, object owner)
{
   return query_deklin(who, art, FALL_DAT, adjektiv, owner);
}
varargs string Wem(mixed who, int art, mixed adjektiv, object owner)
{
   string res;
   return (res = query_deklin(who, art, FALL_DAT, adjektiv, owner)) ?
      capitalize(res) : 0;
}


/*
FUNKTION: wen
DEKLARATION: varargs string wen(mixed who, int art, mixed adjektiv, object owner)
BESCHREIBUNG:
Liefert im Akkusativ deklinierte Form eines Objektnamens.
Dabei wird beruecksichtigt, ob das Objekt ein personal_name hat.
Weitere Beschreibung ->wer
VERWEISE: wer, wem
GRUPPEN: simul_efun, grammatik
*/
varargs string wen(mixed who, int art, mixed adjektiv, object owner)
{
   return query_deklin(who, art, FALL_AKK, adjektiv, owner);
}
varargs string Wen(mixed who, int art, mixed adjektiv, object owner)
{
   string res;
   return (res = query_deklin(who, art, FALL_AKK, adjektiv, owner)) ?
      capitalize(res) : 0;
}


/* ============= User Interface fuer query_deklin ============== */



/*
FUNKTION: der
DEKLARATION: varargs string der(mixed who, mixed adjektiv)
BESCHREIBUNG:
Liefert im Nominativ deklinierte Form eines Objektnamens mit bestimmten Artikel
Dabei wird beruecksichtigt, ob das Objekt ein personal_name hat
(query_personal()==1).

  who   Das Objekt dessen deklinierte Form im Nominativ gewuenscht ist.

	ODER eine virtuelles Objekt in der Form:
	   ([ "name" : "...", "gender" : "..." ]) 
	oder bei Plural
	   ([ "name" : "...", "gender" : "...", "plural": 1 ]) 

  adjektiv  (mixed)
     string  ->  Grundform eines Adjektivs
                 "" bedeutet kein Adjektiv
     string* ->  ({ unregelm.Grundform, Deklinationsstamm }) eines Adjektivs
     int i   ->  i == 0   Alle Adjektive des Objekts who.
                 i >  0   Die ersten i Adjektive des Objekts who.
                 i <  0   Das -i. te Adjektiv des Objekts who.
     int*    ->  ({ a, b }) Die a bis b. Adjektive des Objekts who.

Mit Hilfe von adjektiv koennen alle oder bestimmte Adjektive des Objekts
oder ein fremdes Adjektiv in den Ergebnisstring zwischen Artikel und
Substantiv gestellt werden.
(Adjektive eines Objekts setzt man mit set_adjektiv, siehe dort.)

Alle Parameter sind optional.
Defaultwerte sind dann
   who:      this_object()
   adjektiv: 0  (alle Adjektive des Objekts who)

BEISPIEL:
   fuer  der, dem, den, einen, einem, einen, dein, deinem, deinen,
         sein, seinem, seinen, dieser, diesem, diesen, er, ihm, ihn

   object Spieler, Ork, Gegenstand;
   Spieler    sei eine Person NAMENS Laura  (also mit personal_name)
   Ork        sei ein Ork ohne Namen mit Adjektiv boese (ohne personal_name)
   Gegenstand sei eine Jacke, die Laura bei sich traegt.

   Der(Spieler)+" zieht "+einem(Ork)+" "+seinen(Gegenstand)+" an."
      liefert:
   "Laura zieht einem boesen Ork ihre Jacke an."

   Dieser(Ork,"frech")+" laesst sich das von "+dem(Spieler)+" nicht gefallen."
      liefert:
   "Dieser freche Ork laesst sich das von Laura nicht gefallen."

   Er(Ork)+" haette "+ihn(Spieler)+" am liebsten davongejagt."
      liefert:
   "Er haette sie am liebsten davongejagt."

   Ein(Ork)+" schaut sich neugierig "+
   seinen((["name":"ausruestung",
	    "gender":"weiblich"]),"gut",Spieler)+" an."
      liefert:
   "Ein Ork schaut sich neugierig ihre gute Ausruestung an."

HINWEIS:
   Saemtliche Funktionen existieren nur in MAENNLICHER Form im
   Singular.
   Die Funktionen erzeugen daraus abhaengig vom Geschlecht und
   Zahl des Objekts die RICHTIGE Form.
   Die Funktionen existieren jeweils in Gross- und Kleinschrift.
   ( der und Der, den und Den,... )
ANMERKUNG:
   Keine Diskriminierung beabsichtigt. Die maennlichen Deklinations-
   formen sind die einzigen die sich im Nominativ und Akkusativ
   voneinander unterscheiden, und nur darauf kommt's hier an. :-)

Saemtliche aufgefuehrte Funktionen sind Spezialfaelle von
   wer, wen, wem, Wer, Wen, Wem.


VERWEISE: der, dem, den, einen, einem, einen, dein, deinem, deinen,
          sein, seinem, seinen, dieser, diesem, diesen, er, ihn, ihm,
          Der, Dem, Den, Einen, Einem, Einen, Dein, Deinem, Deinen,
          Sein, Seinem, Seinen, Dieser, Diesem, Diesen, Er, Ihn, Ihm,
          wer, wen, wem, Wer, Wen, Wem,
          set_adjektiv
GRUPPEN: simul_efun, grammatik
*/
varargs string der(mixed w,mixed a)    { return wer(w, ART_DER, a); }
varargs string Der(mixed w,mixed a)    { return Wer(w, ART_DER, a); }

/*
FUNKTION: des
DEKLARATION: varargs string des(mixed who, mixed adjektiv)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe der
VERWEISE: der
GRUPPEN: simul_efun, grammatik
*/
varargs string des(mixed w,mixed a)    { return wessen(w, ART_DER, a); }
varargs string Des(mixed w,mixed a)    { return Wessen(w, ART_DER, a); }

/*
FUNKTION: dem
DEKLARATION: varargs string dem(mixed who, mixed adjektiv)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe der
VERWEISE: der
GRUPPEN: simul_efun, grammatik
*/
varargs string dem(mixed w,mixed a)    { return wem(w, ART_DER, a); }
varargs string Dem(mixed w,mixed a)    { return Wem(w, ART_DER, a); }

/*
FUNKTION: den
DEKLARATION: varargs string den(mixed who, mixed adjektiv)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe der
VERWEISE: der
GRUPPEN: simul_efun, grammatik
*/
varargs string den(mixed w,mixed a)    { return wen(w, ART_DER, a); }
varargs string Den(mixed w,mixed a)    { return Wen(w, ART_DER, a); }

/*
FUNKTION: ein
DEKLARATION: varargs string ein(mixed who, mixed adjektiv)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe der
VERWEISE: der
GRUPPEN: simul_efun, grammatik
*/
varargs string   ein(mixed w,mixed a)  { return wer(w, ART_EIN, a); }
varargs string   Ein(mixed w,mixed a)  { return Wer(w, ART_EIN, a); }

/*
FUNKTION: eines
DEKLARATION: varargs string eines(mixed who, mixed adjektiv)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe der
VERWEISE: der
GRUPPEN: simul_efun, grammatik
*/
varargs string eines(mixed w,mixed a)  { return wessen(w, ART_EIN, a); }
varargs string Eines(mixed w,mixed a)  { return Wessen(w, ART_EIN, a); }

/*
FUNKTION: einem
DEKLARATION: varargs string einem(mixed who, mixed adjektiv)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe der
VERWEISE: der
GRUPPEN: simul_efun, grammatik
*/
varargs string einem(mixed w,mixed a)  { return wem(w, ART_EIN, a); }
varargs string Einem(mixed w,mixed a)  { return Wem(w, ART_EIN, a); }

/*
FUNKTION: einen
DEKLARATION: varargs string einen(mixed who, mixed adjektiv)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe der
VERWEISE: der
GRUPPEN: simul_efun, grammatik
*/
varargs string einen(mixed w,mixed a)  { return wen(w, ART_EIN, a); }
varargs string Einen(mixed w,mixed a)  { return Wen(w, ART_EIN, a); }

/*
FUNKTION: dein
DEKLARATION: varargs string dein(mixed who, mixed adjektiv, object owner)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe der
Zusaetzlicher Parameter owner gibt den Besitzer von who an.
Fehlt owner, dann wird das naechste living environment von who genommen.
Hat who kein living environment, so wird auf automatische Artikelauswahl
(ein/der) umgeschaltet.
VERWEISE: der
GRUPPEN: simul_efun, grammatik
*/
varargs string   dein(mixed w,mixed a,object o) { return wer(w,ART_DEIN,a,o); }
varargs string   Dein(mixed w,mixed a,object o) { return Wer(w,ART_DEIN,a,o); }

/*
FUNKTION: deines
DEKLARATION: varargs string deines(mixed who, mixed adjektiv, object owner)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe der
Zusaetzlicher Parameter owner gibt den Besitzer von who an.
Fehlt owner, dann wird das naechste living environment von who genommen.
Hat who kein living environment, so wird auf automatische Artikelauswahl
(ein/der) umgeschaltet.
VERWEISE: der
GRUPPEN: simul_efun, grammatik
*/
varargs string deines(mixed w,mixed a,object o) {return wessen(w,ART_DEIN,a,o);}
varargs string Deines(mixed w,mixed a,object o) {return Wessen(w,ART_DEIN,a,o);}

/*
FUNKTION: deinem
DEKLARATION: varargs string deinem(mixed who, mixed adjektiv, object owner)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe der
Zusaetzlicher Parameter owner gibt den Besitzer von who an.
Fehlt owner, dann wird das naechste living environment von who genommen.
Hat who kein living environment, so wird auf automatische Artikelauswahl
(ein/der) umgeschaltet.
VERWEISE: der
GRUPPEN: simul_efun, grammatik
*/
varargs string deinem(mixed w,mixed a, object o) { return wem(w,ART_DEIN,a,o);}
varargs string Deinem(mixed w,mixed a, object o) { return Wem(w,ART_DEIN,a,o);}

/*
FUNKTION: deinen
DEKLARATION: varargs string deinen(mixed who, mixed adjektiv, object owner)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe der
Zusaetzlicher Parameter owner gibt den Besitzer von who an.
Fehlt owner, dann wird das naechste living environment von who genommen.
Hat who kein living environment, so wird auf automatische Artikelauswahl
(ein/der) umgeschaltet.
VERWEISE: der
GRUPPEN: simul_efun, grammatik
*/
varargs string deinen(mixed w,mixed a,object o) { return wen(w,ART_DEIN,a,o); }
varargs string Deinen(mixed w,mixed a,object o) { return Wen(w,ART_DEIN,a,o); }

/*
FUNKTION: sein
DEKLARATION: varargs string sein(mixed who, mixed adjektiv, object owner)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe der
Zusaetzlicher Parameter owner gibt den Besitzer von who an.
Fehlt owner, dann wird das naechste living environment von who genommen.
Hat who kein living environment, so wird auf automatische Artikelauswahl
(ein/der) umgeschaltet.
VERWEISE: der
GRUPPEN: simul_efun, grammatik
*/
varargs string sein(mixed w,mixed a,object o) { return wer(w, ART_SEIN, a, o); }
varargs string Sein(mixed w,mixed a,object o) { return Wer(w, ART_SEIN, a, o); }

/*
FUNKTION: seines
DEKLARATION: varargs string seines(mixed who, mixed adjektiv, object owner)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe der
Zusaetzlicher Parameter owner gibt den Besitzer von who an.
Fehlt owner, dann wird das naechste living environment von who genommen.
Hat who kein living environment, so wird auf automatische Artikelauswahl
(eines/des) umgeschaltet.
VERWEISE: der
GRUPPEN: simul_efun, grammatik
*/
varargs string seines(mixed w,mixed a,object o) {return wessen(w,ART_SEIN,a,o);}
varargs string Seines(mixed w,mixed a,object o) {return Wessen(w,ART_SEIN,a,o);}

/*
FUNKTION: seinem
DEKLARATION: varargs string seinem(mixed who, mixed adjektiv, object owner)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe der
Zusaetzlicher Parameter owner gibt den Besitzer von who an.
Fehlt owner, dann wird das naechste living environment von who genommen.
Hat who kein living environment, so wird auf automatische Artikelauswahl
(einem/dem) umgeschaltet.
VERWEISE: der
GRUPPEN: simul_efun, grammatik
*/
varargs string seinem(mixed w,mixed a,object o) { return wem(w, ART_SEIN, a, o); }
varargs string Seinem(mixed w,mixed a,object o) { return Wem(w, ART_SEIN, a, o); }

/*
FUNKTION: seinen
DEKLARATION: varargs string seinen(mixed who, mixed adjektiv, object owner)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe der
Zusaetzlicher Parameter owner gibt den Besitzer von who an.
Fehlt owner, dann wird das naechste living environment von who genommen.
Hat who kein living environment, so wird auf automatische Artikelauswahl
(einen/den) umgeschaltet.
VERWEISE: der
GRUPPEN: simul_efun, grammatik
*/
varargs string seinen(mixed w,mixed a,object o) { return wen(w, ART_SEIN, a, o); }
varargs string Seinen(mixed w,mixed a,object o) { return Wen(w, ART_SEIN, a, o); }

/*
FUNKTION: dieser
DEKLARATION: varargs string dieser(mixed who, mixed adjektiv)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe der
VERWEISE: der
GRUPPEN: simul_efun, grammatik
*/
varargs string dieser(mixed w,mixed a) { return wer(w, ART_DIESER, a); }
varargs string Dieser(mixed w,mixed a) { return Wer(w, ART_DIESER, a); }

/*
FUNKTION: dieses
DEKLARATION: varargs string dieses(mixed who, mixed adjektiv)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe der
VERWEISE: der
GRUPPEN: simul_efun, grammatik
*/
varargs string dieses(mixed w,mixed a) { return wessen(w, ART_DIESER, a); }
varargs string Dieses(mixed w,mixed a) { return Wessen(w, ART_DIESER, a); }

/*
FUNKTION: diesem
DEKLARATION: varargs string diesem(mixed who, mixed adjektiv)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe der
VERWEISE: der
GRUPPEN: simul_efun, grammatik
*/
varargs string diesem(mixed w,mixed a) { return wem(w, ART_DIESER, a); }
varargs string Diesem(mixed w,mixed a) { return Wem(w, ART_DIESER, a); }

/*
FUNKTION: diesen
DEKLARATION: varargs string diesen(mixed who, mixed adjektiv)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe der
VERWEISE: der
GRUPPEN: simul_efun, grammatik
*/
varargs string diesen(mixed w,mixed a) { return wen(w, ART_DIESER, a); }
varargs string Diesen(mixed w,mixed a) { return Wen(w, ART_DIESER, a); }

/*
FUNKTION: er
DEKLARATION: varargs string er(mixed who)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe der
Anders als bei der fehlt der Parameter adjektiv, Personalpronomina
haben keine Adjektive.
VERWEISE: der
GRUPPEN: simul_efun, grammatik
*/
varargs string  er(mixed w)            { return wer(w, ART_ER); }
varargs string  Er(mixed w)            { return Wer(w, ART_ER); }

/*
FUNKTION: ihm
DEKLARATION: varargs string ihm(mixed who)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe der
Anders als bei der fehlt der Parameter adjektiv, Personalpronomina
haben keine Adjektive.
VERWEISE: der
GRUPPEN: simul_efun, grammatik
*/
varargs string ihm(mixed w)            { return wem(w, ART_ER); }
varargs string Ihm(mixed w)            { return Wem(w, ART_ER); }

/*
FUNKTION: ihn
DEKLARATION: varargs string ihn(mixed who)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe der
Anders als bei der fehlt der Parameter adjektiv, Personalpronomina
haben keine Adjektive.
VERWEISE: der
GRUPPEN: simul_efun, grammatik
*/
varargs string ihn(mixed w)            { return wen(w, ART_ER); }
varargs string Ihn(mixed w)            { return Wen(w ,ART_ER); }



/* ========= User Interface fuer query_deklin_owner ============ */


/*
FUNKTION: ihr
DEKLARATION: varargs string ihr(mixed who, mixed a1, mixed owner, mixed a2)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe der

Der zusaetzlicher Parameter owner gibt den Besitzer von who an.
Fehlt owner, dann wird das naechste living environment von who genommen.
Hat who kein living environment, so wird auf automatische Artikelauswahl
(ein/der) umgeschaltet.
***WICHTIG***:
ihr() liefert NICHT 'ihr Testobjekt' (das macht ja sein() schon automatisch),
SONDERN 'Stellas Testobjekt'. Also NICHT das POSSESIVPRONOMEN, sondern
den NAMEN des Besitzers selbst. Ist der Besitzer nicht 'personal', dann
liefert ihr() beispielsweise: 'das Testobjekt des Orks'.

Der zusaetzliche Parameter a2:
-  a2 ist die Adjektivspezifikation fuer den BESITZER owner
   (analog zu a1, welches die Adjektive des OBJEKTS who spezifiziert)

VERWEISE: der
GRUPPEN: simul_efun, grammatik
*/
varargs string ihr(mixed w, mixed a1, mixed o, mixed a2, 
		   string name, int oart, int wart)
{
   if(!oart)
      oart = ART_AAA;
   return query_deklin_owner(w, wart, FALL_NOM, a1, o, oart, a2, name);
}
varargs string Ihr(mixed w, mixed a1, mixed o, mixed a2,
		   string name, int oart, int wart)
{
   string res;
   return (res = ihr(w, a1, o, a2, name, oart, wart)) ? capitalize(res) : 0;
}

/*
FUNKTION: ihrem
DEKLARATION: varargs string ihrem(mixed w, mixed a1, mixed o, mixed a2, string name)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe ihr
VERWEISE: ihr, sein, der
GRUPPEN: simul_efun, grammatik
*/
varargs string ihrem(mixed w, mixed a1, mixed o, mixed a2, 
		     string name, int oart, int wart)
{
   if(!oart)
      oart = ART_AAA;
   return query_deklin_owner(w, wart, FALL_DAT, a1, o, oart, a2, name);
}
varargs string Ihrem(mixed w, mixed a1, mixed o, mixed a2,
		   string name, int oart, int wart)
{
   string res;
   return (res = ihrem(w, a1, o, a2, name, oart, wart)) ? capitalize(res) : 0;
}

/*
FUNKTION: ihren
DEKLARATION: varargs string ihren(mixed w, mixed a1, mixed o, mixed a2, string name)
BESCHREIBUNG:
Liefert deklinierte Form eines Objektnamens.   siehe ihr
VERWEISE: ihr, sein, der
GRUPPEN: simul_efun, grammatik
*/
varargs string ihren(mixed w, mixed a1, mixed o, mixed a2, 
		     string name, int oart, int wart)
{
   if(!oart)
      oart = ART_AAA;
   return query_deklin_owner(w, wart, FALL_AKK, a1, o, oart, a2, name);
}
varargs string Ihren(mixed w, mixed a1, mixed o, mixed a2,
		   string name, int oart, int wart)
{
   string res;
   return (res = ihren(w, a1, o, a2, name, oart, wart)) ? capitalize(res) : 0;
}

/* ===================== end of deklin ========================= */
