/*******************
** Eldarea MUDLib **
********************
**
** std/shells/filesys/filesys.c -- filesystem command handling
**
** CVS DATA
** $Date: 2000/12/04 11:26:06 $
** $Revision: 1.2 $
**
** CVS History
**
** $Log: filesys.c,v $
** Revision 1.2  2000/12/04 11:26:06  elatar
** new color handling adapted
** .readme files are viewed automatically when changing to directory
** crontab command implemented
**
** Revision 1.1.1.1  1999/11/05 12:30:47  elatar
** Preparing mudlib for cvs control
**
**
*/

#pragma strong_types

private inherit "/std/shells/filesys/manual";
private inherit "/std/shells/filesys/primitives";
private inherit "/std/shells/filesys/asynchron";

#define NEED_PROTOTYPES

#include <thing/properties.h>
#include <properties.h>
#include <defines.h>
#include <config.h>
#include <moving.h>
#include <wizlevels.h>
#include <homes.h>
#include <telnet.h>
#include <udp.h>
#include <cron.h>
#include <color.h>

#undef NAME // udp und sys_debug benutzen NAME, udps Version hier unnuetz
#include "/std/sys_debug.h"
#define C	(LFUN|CALL|RET|ARGS|APPLY)
#define MAND "/global/service/manbuf"

#define GET_PATH	(symbol_function("_get_path", find_object(MASTER)))
#define USAGE(str)	(notify_fail(sprintf("usage: %s\n", str)), 0)

private string *_ed_stack = ({});	// ed/more file stack (item 0 contains
                                  // function to call

// prototypes
static int _prompt(string prompt);
static int _cd(string path);

private string _set_last_directory(string dir);
private string _set_currentdir(string path);
private string _query_currentdir();

static mixed *_query_localcmds()
{
  return ({
          ({"ls","_ls",0,LEARNER_LVL}),
	  ({"cd","_cd",0,LEARNER_LVL}),
	  ({"pwd","_pwd",0,LEARNER_LVL}),
	  ({"prompt","_prompt",0,LEARNER_LVL}),
	  ({"cp","_cp",0,LEARNER_LVL}),
	  ({"mv","_mv",0,LEARNER_LVL}),
	  ({"rm","_rm",0,LEARNER_LVL}),
	  ({"mkdir","_mkdir",0,LEARNER_LVL}),
	  ({"rmdir","_mkdir",0,LEARNER_LVL}),
	  ({"cat","_cat",0,LEARNER_LVL}),
	  ({"head","_cat",0,LEARNER_LVL}),
	  ({"tail","_cat",0,LEARNER_LVL}),
	  ({"man","_man",0,LEARNER_LVL}),
	  ({"more","_edit",0,LEARNER_LVL}),
	  ({"ed","_edit",0,LEARNER_LVL}),
	  ({"grep","_grep",0,LEARNER_LVL}),
	  ({"upd","_upd",0,LEARNER_LVL}),
	  ({"crontab","_crontab",0,LEARNER_LVL})
         });
}

void initialize()
{
  Set(P_CURRENTDIR,#'_set_currentdir,F_SET_METHOD);
  Set(P_LAST_DIRECTORY,SAVE,F_MODE);
  Set(P_LAST_DIRECTORY,#'_set_last_directory,F_SET_METHOD);
  _prompt(QueryProp(P_PROMPT));
  /* cd to home directory, if existing */
  _cd((string)0);
}

// base() -- returns the basename of the file
private string base(string file) 
{ string *tmp; return sizeof(tmp = explode(file, "/"))?tmp[<1]:""; }

// path() -- returns the path part of the filename
private string path(string file) 
{ return (file[0]=='/'?"/":"")+implode(explode(file,"/")[0..<2],"/"); }

// normalize() -- gives the full path and name of the file
// additionas arguments: the current path (__path) and the normalization
// function (usually _get_path() in master.c) (__norm)
// normalizes only basenames!! /pl*/ha* will not be normalized
private string normalize(string file, 
                         string __path, closure __norm)
{ return funcall(__norm, (file[0]=='/'||file[0]=='~'||
			  file[0]=='+'||file[0]=='$'?
                          file:__path+file), getuid(ME)); }

// _glob() -- globbing a string
// additional arguments: the resulting command line array (line) to be given
// by reference, the current path (__path) and a normalization function (__norm)
private mixed _glob(string str, string *line, string __path, closure __norm)
{
  string *tmp, p;
  while(str[0] == ' ') str = str[1..];
  if(str[0] == '"') { line += ({ str[1..<2] }); return; } // preserve strings
  if(str[0] == '-') { line += ({ str }); return; }	  // preserve flags
  if((member(str, '*') != -1 || member(str, '?') != -1) &&
     (tmp = get_dir(p = normalize(str, __path, __norm))))
  {
    if(base(str)[0] != '.') tmp = regexp(tmp, "^[^.].*");
    line += map_array(tmp, lambda(({'s}), ({#'+, path(p)+"/", 's})));
  }
  else 
    line += ({ normalize(str, __path, __norm) });
}

// glob() -- command line globbing
// takes the commandline as argument and returns an array with all semantic
// parts
string *glob(string cmdline)
{
  int i;
  string *line; line = ({});
  map_array(regexplode(cmdline, "[\"][^\"]*[\"]| -[a-zA-Z][a-zA-Z]*| ")
           -({" ", ""}),
            #'_glob, &line, QueryProp(P_CURRENTDIR)+"/", GET_PATH);
  return line - ({""});
}

// GetOpt() -- get all options (-blafu)
// takes as first arguement an array containing strings like "-xyz", the
// possible option characters as second and the command calling as third.
// returns an integer with bits set when matching option is found
private int GetOpt(string *cmdline, string opts, string __cmd)
{
  int flags;
  while(sizeof(cmdline) && cmdline[0][0] == '-')
  {
    int i, f; i = strlen(cmdline[0]);
    while(i-- && cmdline[0][i] != '-')
      if((f = member(opts, cmdline[0][i])) != -1) 
        flags |= (1 << f);
      else return (printf("%s: illegal flag '%c'\n", __cmd, cmdline[0][i]), -1);
    cmdline = cmdline[1..];
  }
  return flags;
}

// ls command
#define LS_A	0x01			// list all files
#define LS_L	0x02			// long listing
#define LS_C    0x04                    // toggle color enhancement
#define LS_OPTS		"alc"

// strlcmp() -- compares the length of two strings (returns s1 < s2)
private int strlcmp(string s1, string s2) { return strlen(s1) < strlen(s2); }

// add_slash() -- adds a slash to directories and * to loaded blueprints
private string add_slash(string file, string p) 
{
  if(file_size((file[0]=='/'?"":p)+file)==-2)
    return file+"/";
  if(objectp(find_object((file[0]=='/'?"":p)+file)))
    return file+"*";
  return file;
}

// _ls_output() -- creates output from files given
// if file is a dir directories is extended by the listed dir
// if LS_A flag is set, also .* files will be listed
private mixed _ls_output(string file, string directories, int __flags)
{
  string *files;
  mixed tmp;
  if(files = get_dir(file+"/*"))
  { int tab;
    object vc;
    if(vc = find_object(file+"/virtual_compiler"))
      if(call_resolved(&tmp, vc, "QueryObjects") && pointerp(tmp))
        files += map_array(tmp, lambda(({'o}), 
                                       ({#'base, ({#'file_name, 'o}) })));
      else printf("ls: /%O is not a standard virtual compiler\n", vc);
    if(!(__flags & LS_A)) files = regexp(files, "^[^.].*");
    files = map_array(files, #'add_slash, file+"/");
    if(sizeof(files)) tab = strlen(sort_array(files, #'strlcmp)[0])+1;
    directories += sprintf("%s:   (%d files/directories)\n",file,sizeof(files))
                 + break_string(sprintf("%-"+tab+"@s", files), 78) + "\n";
    return 0;
  }
  return base(add_slash(file, ""));
}

// _long() -- returns a long (LS_L) listing of the directory
private void _long(string directory, int flags, closure __get_dir)
{ write(funcall(__get_dir, directory)); }

// _ls() -- list directories
static int _ls(string args)
{
  int i, j, flags;
  string *pargs, *files, directories;
  mapping cols;
  if(!stringp(args)) args = QueryProp(P_CURRENTDIR);
  if(!sizeof(pargs = glob(args))) return USAGE("ls [-alc] filename");

  if((flags = GetOpt(&pargs, LS_OPTS, "ls")) == -1) return 1;

  if(!sizeof(pargs)) 
    pargs = ({ normalize(".", QueryProp(P_CURRENTDIR)+"/", GET_PATH) });

  if (flags & LS_C) {
    cols=QueryProp(P_COLORS);
    if (!member(cols, "ls-dir")) // dann einschalten auf default
      cols+=(["ls-dir": 4, "ls-loaded": 1, "ls-c": 5, "ls-h": 2 ]);
    else cols-=(["ls-dir","ls-loaded","ls-c","ls-h"]);
    SetProp(P_COLORS, cols);
  }
  if(flags & LS_L)
  {
    map_array(pargs, #'_long, flags,
                     symbol_function("show_dir", find_object(MASTER)));
    return 1;
  }
  directories = "";
  files = map_array(pargs, #'_ls_output, &directories, flags) - ({0});
  if(sizeof(files))
  { int tab;
    if(!(flags & LS_A)) files = regexp(files, "^[^.].*");
    if(sizeof(files)) tab = strlen(sort_array(files, #'strlcmp)[0])+1;
    directories = break_string(sprintf("%-"+tab+"@s", files), 78)
                + "\n"+directories;
  }
  directories = implode(regexp(regexplode(directories, "\n[ ][ ]*"), 
                       "^[^\n][^ ].*"), "\n");
  // frag mich keiner, was das hier drueber machen soll...

  // Einbau der Farben
  files=regexplode(directories, "[ \n][ \n]*");
  directories="";
  for (i=0, j=sizeof(files)-1; i<j; i+=2) {
    switch(files[i][<1]) {
      case '/': directories+=colorize(files[i], CG_LS_DIR); break;
      case '*': directories+=colorize(files[i], CG_LS_LOADED); break;
      case 'c': if (files[i][<2]=='.') {
                  directories+=colorize(files[i], CG_LS_C); break;
                }
      case 'h': if (files[i][<2]=='.') {
                  directories+=colorize(files[i], CG_LS_H); break;
                }
      default:   directories+=files[i];
    }
    directories+=files[i+1];
  }
                
  write(directories);
  return 1;
}

// __lstrip(string) -- strips spaces from string
private static string __lstrip(string arg)
{
  return implode(efun::explode(arg," "),"");
}

// _cd() -- change current directory
static int _cd(string args)
{
  string *pargs, msg;
  int flags;
  if(!stringp(args) || !strlen(args)) args = "/players/"+getuid(ME);
  if(__lstrip(args)=="-") args = QueryProp(P_LAST_DIRECTORY);
  if(!sizeof(pargs = glob(args)))
    USAGE("cd [-ls] directory");
  
  if(flags = GetOpt(&pargs, "ls", "cd"))
  {
    if(flags & 1) SetProp(P_SHORT_CWD, 0);
    if(flags & 2) SetProp(P_SHORT_CWD, 1);
    write("Directory status: "+(QueryProp(P_SHORT_CWD)?"short":"long")+"\n");
  }
  if(!sizeof(pargs)) pargs = ({ "/players/"+getuid(ME) });
  switch(file_size(pargs[0]))
  {
  case -2: break;
  case -1: return (printf("%s: no such file or directory\n", args), 1); break;
  default: return (printf("%s: not a directory\n", args), 1); break;
  }
  SetProp(P_LAST_DIRECTORY,QueryProp(P_CURRENTDIR));
  write(SetProp(P_CURRENTDIR, pargs[0])+"\n");
  if(!QueryProp(P_SHORT_CWD) && file_size(pargs[0] + "/.readme") > 0)
    write((msg = read_file(pargs[0] + "/.readme"))?msg:"");
  return 1;
}

// _pwd() -- print current directory
static int _pwd()
{
  write(QueryProp(P_CURRENTDIR)+"\n");
  return 1;
}

// prompt command

// _subst_prompt() -- substitute prompt macros
// takes a string and returns matching macro value
private string _subst_prompt(string str)
{
  switch(str)
  {
  case "\\h": return strlen(MUDNAME)?MUDNAME:"NamenlosMud";
  case "\\u": return CAP(getuid(ME));
  default: return str;
  }
}

// _prompt() -- change the current prompt
static int _prompt(string args)
{
  string *pargs;
  if(!stringp(args) || !strlen(args)) args = "> ";
  if(args[0]=='"') args = args[1..<2];
  if(!sizeof(pargs = regexplode(args, "\\\\[huw]"))) 
    USAGE("prompt \"<newprompt>\"");

  pargs = map_array(pargs, #'_subst_prompt);
  SetProp(P_PROMPT, implode(pargs, ""));
  _set_currentdir(_query_currentdir());
  return 1;
}
    
// _cp() -- copy files 
#define CP_I		1		// copy interactive
#define CP_R		2		// copy recursively
#define CP_P		4		// preserver file status
#define CP_OPTS		"irp"	

static int _cp(string args)
{
  int flags;
  string *pargs, to;
  closure exec_cmd, qask_cmd;

  if(!stringp(args) || !sizeof(pargs = glob(args)))
    return USAGE("cp [-"CP_OPTS"] f1 f2; or: cp [-"CP_OPTS"] f1 ... fn d2");

  if((flags = GetOpt(&pargs, CP_OPTS, "cp")) == -1) return 1;
  if(flags & CP_I) return (printf("cp: interactive not implemented\n"), 1);
  if(flags & CP_R) return (printf("cp: recursive not implemented\n"), 1);
  if(flags & CP_P) return (printf("cp: preserve not implemented\n"), 1);

  if(sizeof(pargs)<2) return (printf("cp: missing files\n"), 1);
  if(file_size(to = pargs[<1]) == -2)
    exec_cmd = lambda(({'f}), ({#'cp_file,'f,({#'+,to+"/",({#'base, 'f})})}));
  else 
    exec_cmd = #'cp_file;

  asynchron(pargs[0..<2], exec_cmd, ({ to }));
  return 1;
}

// _mv() -- move files
#define MV_I            1		// move interactively
#define MV_F            2		// force overwriting
#define MV_OPTS         "if"

static int _mv(string args)
{
  int flags, pos;
  string *pargs, to;
  closure exec_cmd;

  if(!stringp(args) || !sizeof(pargs = glob(args)))
    return USAGE("mv [-"MV_OPTS"] f1 f2 or mv [-"MV_OPTS"] f1 ... fn d1 "
                +"(`fn' is a file or directory)");

  if((flags = GetOpt(&pargs, MV_OPTS, "mv")) == -1) return 1;
  if(flags & MV_I) return (printf("mv: interactive not implemented\n"), 1);
  if(flags & MV_F) return (printf("mv: force not implemented\n"), 1);

  if(sizeof(pargs)<2) return (printf("mv: missing files\n"), 1);
  if(file_size(to = pargs[<1]) == -2)
    exec_cmd = lambda(({'f}), ({#'mv_file,'f,({#'+,to+"/",({#'base, 'f})})}));
  else
    exec_cmd = #'mv_file;

  while((pos = member(pargs = pargs[0..<2], to)) != -1)
  {
    printf("mv: %s: invalid argument\n", pargs[pos]);
    pargs[pos..pos] = ({});
  }
  asynchron(pargs, exec_cmd, ({ to }));
  return 1;
}

// _rm() -- remove files
#define RM_R		1		// remove recursively
#define RM_I		2		// remove interactively
#define RM_F		4		// force remove (not even interactive)
#define RM_OPTS         "rif"

static int _rm(string args)
{
  int flags, *ret;
  string *pargs, to;
  closure exec_cmd;

  if(!stringp(args) || !sizeof(pargs = glob(args)))
    return USAGE("rm [-"+RM_OPTS+"] file ...");

  if((flags = GetOpt(&pargs, RM_OPTS, "rm")) == -1) return 1;
  if(flags & RM_R) return (printf("rm: recursive not implemented\n"), 1);
  if(flags & RM_I) return (printf("rm: interactive not implemented\n"), 1);
  if(flags & RM_F) return (printf("rm: force not implemented\n"), 1);

  if(!sizeof(pargs)) printf("rm: missing files\n");
  asynchron(pargs, #'rm_file);
  return 1;
}

// _mkdir() -- make or remove directories 
static int _mkdir(string args)
{
  string *pargs, msg;
  if(!stringp(args) || !strlen(args) ||!sizeof(pargs = glob(args))) 
    USAGE((query_verb()=="mkdir"?"mkdir":"rmdir")+" directory ...");

  asynchron(pargs, #'dir_file, ({ query_verb()=="mkdir" }));
  
  return 1;
}

// _cat() -- cat, head or tail a file
static int _cat(string args)
{
  string *pargs, verb;
  int size;
  verb = query_verb();
  if(!stringp(args) || !(size = sizeof(pargs = glob(args))) || size > 1)
    return USAGE(verb+" file");

  switch(file_size(pargs[0]))
  {
  case -2: return (printf("%s: is a directory\n", args), 1); break;
  case -1: return (printf("%s: no such file\n", args), 1); break;
  }
  switch(verb)
  {
  case "cat" : if(!cat(pargs[0])) printf("cat: failed\n"); break;
  case "head": if(!cat(pargs[0], 0, 10)) printf("head: failed\n"); break;
  case "tail": tail(pargs[0]); break;
  }
  return 1;
}

// man command

private int remote_man(string key, string mud) {
  object zettel;
  string x;
  zettel=clone_object("/global/service/manhelp");
  x=INETD->send_udp(mud, ([ REQUEST: "man",
                            SENDER:  zettel,
                            DATA:    key ]), 1);
  if (x) write(x);
  write("Anfrage nach \'"+key+"\' an "+(mud?capitalize(mud):"alle bekannten "
    "Muds")+" abgeschickt.\nEin magischer Zettel flattert Dir zu.\n");
  zettel->SetKey(key);
  zettel->move(this_player());
  return 1;
}

static int _man(string args) {
  // USAGE("man [-i] [-F] [-M path] [-r mud] [-R] word");
  string *pargs, *tmp, manpath, section, mudname;
  int inter, remote, fn;
  if (!args) args="";
  pargs=regexplode(" "+args+" ", " -M [^ ]*| -i | -F | -R | -r [^ ]*| ")-
    ({"", " "});

  manpath="/doc";
  if (member(pargs, " -i ") != -1) {
    write("-i wird ignoriert, da nicht eingebaut.\n");
    pargs -= ({ " -i " });
    // manpath="/htmldoc"; inter=1;
  }
  if (member(pargs, " -F ") != -1) {
    pargs -= ({ " -F " }); fn=1;
  }
  if (member(pargs, " -R ") != -1) {
    pargs -= ({ " -R " }); remote=1;
  }
  if (sizeof(tmp=regexp(pargs, " -M *"))) {
    if (strlen(tmp[0])>4) {
      pargs -= tmp;
      manpath=normalize(tmp[0][4..], QueryProp(P_CURRENTDIR), GET_PATH);
      if (file_size(manpath)!=-2) {
        printf("\'"+manpath+"\' ist kein gueltiger Pfad.\n");
        return 1;
      }
    }
    else return (printf("Pfadangabe fuer Option -M fehlt\n"), 1);
  }
  if (sizeof(tmp=regexp(pargs, " -r *"))) {
    if (strlen(tmp[0])>4) {
      remote=1;
      mudname=tmp[0][4..];
      pargs -= tmp;
    }
    else return (printf("Mudname fuer Option -r fehlt\n"), 1);
  }
  if (remote) {
    if (!sizeof(pargs)) write("Stichwort fuer Anfrage fehlt.\n");
    else remote_man(pargs[0], mudname);
    return 1;
  }

  if (!sizeof(pargs)) {
    write("Hauptstichwoerter:\n"+break_string(implode(
      filter_array(get_dir(manpath+"/*"), lambda( ({'x}),
      ({#'!=, ({ #'[, 'x, 0 }), '.' }) )), ", "), 78));
    return 1;
  }
  if (inter) {
    Manual(pargs[0], manpath);
    return 1;
  }
  else MAND->query_man(manpath, &pargs, 0, fn);
  if (sizeof(pargs))
    write("Nichts passendes fuer "+implode(pargs, ", ")+" gefunden.\n");
  return 1;
}

// grep command
#define GREP_C		1		// count only found lines
#define GREP_H		2		// do not display file names
#define GREP_I		4		// ignore case
#define GREP_L		8		// display explicitely file names
#define GREP_N		16		// enumerate lines
#define GREP_V		32		// print out lines in reverse order
#define GREP_OPTS	"chilnv"

// _do_grep() -- grep in a given file for a given pattern
// takes the file, a regular expression and flags as arguments
private void _do_grep(string file, string rexpr, int flags)
{
  string *result, msg; msg = "";
  switch(file_size(file))
  {
  case 0:
  case -2: return;
  case -1: return printf("%s: no such file or directory\n", file); break;
  }
  if(!pointerp(result = grep_file(file, rexpr, flags)) ||
     !sizeof(result)) return;
  if(flags & GREP_L) return write(file+"\n");
  if(!(flags & GREP_H)) msg += file+":\n";
  if(flags & GREP_C) return write(msg + sizeof(result)+"\n");
  write(msg+implode(result, "\n")+"\n");
}

static int _grep(string args)
{
  int flags, pos;
  string *pargs, rexpr, *tmp;

  if(!stringp(args))
    return USAGE("grep [-"+GREP_OPTS+"] expression file ...");

  pargs = regexplode(args, "-["+GREP_OPTS+"]*")-({"", " "});
  if((flags = GetOpt(&pargs, GREP_OPTS, "grep")) == -1) return 1;
  if(!sizeof(pargs)) return (printf("grep: missing expression\n"), 1);
  tmp = regexplode(pargs[0], "\"[^\"]*\"|[ ]")-({"", " "});
  rexpr = tmp[0]; if(rexpr[0]=='"') rexpr = rexpr[1..<2];
  pargs = glob(implode(tmp[1..], " "));

  if(!sizeof(pargs)) printf("grep: missing files\n");
  if(flags & GREP_I) rexpr = lower_case(rexpr);
  asynchron(pargs, #'_do_grep, ({ rexpr, flags }));
  return 1;
}

// ed/more command
// edit_file() -- edit/more the first file on _ed_stack
static void edit_file()
{
  mixed funct;
  if(sizeof(_ed_stack) > 1) 
  {
    string tmp; tmp = _ed_stack[1]; _ed_stack[1..1] = ({});
    printf("Next file: %s:\n", tmp);
    funct = _ed_stack[0];
    if(funct == #'ed) funcall(_ed_stack[0], tmp, "edit_file");
    else funcall(_ed_stack[0], tmp, 1, #'edit_file);
  }
  else return printf("No more files!\n");
}

// _edit() -- edit/more files
static int _edit(string args)
{
  string *pargs, verb;
  int *size, i;
  verb = query_verb();
  if(!stringp(args) || !sizeof(pargs = glob(args)))
    return USAGE(verb+" file ...");

  size = map_array(pargs, #'file_size);
  while((i = member(size, -2)) != -1) 
  {
    printf("%s: %s is a directory\n", verb, pargs[i]);
    pargs[i..i] = ({}); size[i..i] = ({});
  }
  _ed_stack = ({ verb == "ed" ? #'ed : symbol_function("More", ME) }) + pargs;
  edit_file();
  return 1;
}

// upd command
#define UPD_A		1		// update all found instances
#define UPD_F		2		// find all instances
#define UPD_I		4		// update inherited blueprints
#define UPD_L		8		// update/load files (if not loaded)
#define UPD_R		16		// update/reload files (only loaded)
#define UPD_OPTS	"afilr"

// save_inv() -- move all interactives to DEFAULT_HOME found in o
private mixed save_inv(object o)
{ if(query_once_interactive(o)) 
  { o->move(DEFAULT_HOME, M_GO|M_NO_SHOW|M_NO_ATTACK|M_SILENT);
    return o;
  }
  return 0;
}

// _instance_upd() -- update all instances found for a class
private void _instance_upd(string file, int flags, mixed o)
{
  object tmp, next;
  string name, err;
  int instance;
  while(get_eval_cost() > 80000 &&         // eval cost check
        objectp(o))
	{
    next = debug_info(2, o);
    if(sscanf(file_name(o), "%s#%d", name, instance) == 2 && 
       name == file)
		{
      object *inv, obj;
      tmp = environment(o);
      if(flags & UPD_F)
        printf("upd: /%O found in %s\n", o, tmp?file_name(tmp):"no env");
      if(flags & UPD_A)
      {
        string msg;
        inv = map_array(deep_inventory(o), #'save_inv) - ({0});
        msg = file_name(o)+": remove";
        o->remove();
        if(objectp(o)) { destruct(o); msg += " error, hard destruct"; }
        if(flags & UPD_R)
			  {
          if(!(err = catch(obj = clone_object(file))) && objectp(tmp))
          {
            if(obj->move(tmp, M_GO|M_NO_SHOW|M_SILENT|M_NO_ATTACK)>0)
              map_objects(inv-({0}),"move",obj,
                                    M_GO|M_SILENT|M_NO_SHOW|M_NO_ATTACK);
            else
              printf("upd: /%O: could not move back to /%O\n", obj, tmp);
            msg += ", reclone";
          }
          else
            printf("upd: %s: %s", (objectp(obj)?file_name(obj):file),
                                  (stringp(err)?err:"possible missing env\n"));
        }
        printf(msg+"\n");
      }
    }
    o = next;
  }
  if(objectp(o)) call_out(#'_instance_upd, 0, file, flags, o);
  else printf("upd: %s: all instances %s\n", file,
                                            (flags & UPD_A)?"updated":"found");
}

// _do_upd() -- does the update
// takes a file and flags as arguments
private void _do_upd(string file, int flags)
{
  string msg, err;
  object obj; msg = "";

  // Geaendert von Serpeniel, 08.04.1998
  if(file[<2..]!=".c")
     file+=".c";

  // Geaendert von Serpeniel, 08.04.1998
  // if(file[<2..]!=".c" || file == DEFAULT_HOME)
  if(file==DEFAULT_HOME) 
    return printf("upd: %s: ignored (not loadable)\n", file);
  if(file[0]!='/') file = "/"+file;
  if(flags & UPD_F || flags & UPD_A)
	{
    if ((flags & UPD_A) && upd_file(file, 1) < 1) return;
    call_out(#'_instance_upd, 0, file[0..<3], flags, debug_info(2,0));
    if(flags & UPD_F && !(flags & (UPD_R | UPD_L))) return;
  }
  if(objectp(obj = find_object(file)))
  {
    object *inv;
    string *ilist; ilist = inherit_list(obj);
    inv = map_array(deep_inventory(obj), #'save_inv) - ({0});
    if(upd_file(file) < 0) return;
    msg += "update";
    if(flags & UPD_I) 
      map_array(ilist, #'_do_upd, flags & ~(UPD_I|UPD_L|UPD_R|UPD_F|UPD_A));
    if((flags & UPD_R) || (flags & UPD_L))
      if(!(err = catch(call_other(file, "?"))))
      {
        msg += ", reload";
        map_objects(inv-({0}),"move",file,M_GO|M_SILENT|M_NO_SHOW|M_NO_ATTACK);
      }
    else 
      return printf("%s: %s", file, err);
  }
  else 
    if(!upd_file(file))
      if(flags & UPD_L)
        if(err = catch(call_other(file, "?")))
          return printf("%s: %s", file, err);
        else msg += "load";
  if(strlen(msg)) write(file+": "+msg+"\n");
}

// _upd() -- update files
static int _upd(string args)
{
  int flags, *ret;
  string *pargs;

  if(!stringp(args) || !sizeof(pargs = glob(args)))
    return USAGE("upd [-"+UPD_OPTS+"] file ...");

  if((flags = GetOpt(&pargs, UPD_OPTS, "upd")) == -1) return 1;
  if((flags & UPD_A || (flags & UPD_F && flags != UPD_F)) && !IS_ELDER(ME)) 
    return (printf("upd: -f and -a flag only for elder use\n"), 1);

  if(!sizeof(pargs)) printf("upd: missing files\n");
  asynchron(pargs, #'_do_upd, ({ flags }));
  return 1;
}

static string _crontab_usage()
{
  return ("usage:  crontab [-u user] file\n"
         "         crontab [-u user] { -e | -l | -r }\n"
         "         (default operation is replace)\n"
         "-e       (edit user's crontab)\n"
         "-l       (list user's crontab)\n"
         "-r       (remove user's crontab)\n");
}

static int _crontab(string args)
{
  string * pargs;
  
  if (!stringp(args) || sizeof(pargs=explode(args," "))==0)  
  {
    notify_fail(
      "crontab: usage error: file name must be specified for replace\n"
      +_crontab_usage());
    return 0;
  }
  
  switch(CRON->cmd_crontab(pargs))
  {
    case 0:
      return 0;
    case -1:
      notify_fail(
        "crontab: usage error: option -u requires argument\n"
        +_crontab_usage());
      return 0;
      break;
    case -2:
      notify_fail(
        "crontab: usage error: only one operation permitted\n"
        +_crontab_usage());
      return 0;
    case -3:
      notify_fail(
        "crontab: usage error: file name must be specified for replace\n"
        +_crontab_usage());
      return 0;
    case -4:
      notify_fail(
        "crontab: usage error: unrecognized option\n"
        +_crontab_usage());
      return 0;
    default:
      return 1;      
  }
}

// property methods
private string _set_currentdir(string path) {
  mapping TN;
  string promptstring; 
  if (!(promptstring = QueryProp(P_PROMPT)) || promptstring=="")  
    promptstring="> ";
  else promptstring=implode(efun::explode(promptstring,"\\w"), path); 
  TN = query_telnet_neg();
 
  if (mappingp(TN) && mappingp(TN["received"]) &&
    (TN["received"][TELOPT_EOR,1] == DO)) set_prompt(lambda( ({}),
      ({ #'efun::binary_message,
         ({ #'+, ({ #'to_array, ({ #'implode, ({ #'efun::explode,
                    promptstring, "\\t" }), ({ #'[..],
                    ({ #'ctime }), 11, 15 }) }) }),
                 ({ #'({, IAC, EOR }) }) })), this_object());
  else set_prompt(lambda( ({}),
      ({ #'implode, ({ #'efun::explode, promptstring, "\\t" }),
                    ({ #'[..], ({ #'ctime }), 11, 15 })
      })), this_object());
 
  return Set(P_CURRENTDIR, path);
}

private string _query_currentdir()
{
  string c;
  if(!c=Query(P_CURRENTDIR)) return Set(P_CURRENTDIR, "/players/"+getuid(ME));
  return c;
}

private string _query_last_directory()
{
  string c;
  if(!c=Query(P_LAST_DIRECTORY)) 
    return Set(P_LAST_DIRECTORY, "/players/"+getuid(ME));
  return c;
}

private string _set_last_directory(string dir)
{
  string c;
  if(!stringp(dir) || !strlen(dir)) return 0;
  if(file_size((c="/secure/master"->make_path_absolute(dir)))!=-2)
    return 0;
  return Set(P_LAST_DIRECTORY,c);
}

// damit telnetneg.c den Prompt auch von Ausserhalb setzen kann
void renew_prompt()
{
  _set_currentdir(_query_currentdir());
}


static void reconnect()
{
  _cd(QueryProp(P_CURRENTDIR));
}

// TO BE REMOVED

mixed find_file(string str, string suffix)
{
  string res;
  res = normalize(str+(stringp(suffix)?suffix:""), 
                  QueryProp(P_CURRENTDIR)+"/", GET_PATH);
  if(file_size(res)>=0) return res;
}

